package xray

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestRoundTrip(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	rt := x.RoundTripper(http.DefaultTransport)

	var responseContentLength int
	var headers xRayHeaders
	ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		headers = parseHeadersForTest(r.Header)

		b := []byte(`200 - Nothing to see`)
		responseContentLength = len(b)
		w.WriteHeader(http.StatusOK)
		_, err := w.Write(b)
		assert.NoError(t, err)
	}))

	defer ts.Close()

	reader := strings.NewReader("")
	req := httptest.NewRequest("GET", ts.URL, reader)

	_, err := rt.RoundTrip(req)
	assert.NoError(t, err)

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, "remote", s.Namespace)
	assert.Equal(t, "GET", s.HTTP.Request.Method)
	assert.Equal(t, ts.URL, s.HTTP.Request.URL)
	assert.Equal(t, 200, s.HTTP.Response.Status)
	assert.Equal(t, responseContentLength, s.HTTP.Response.ContentLength)
	assert.Equal(t, headers.RootTraceID, s.TraceID)
	assert.Equal(t, headers.ParentID, s.ID)
	//not checking for sampling here
}

func TestBadRoundTrip(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	reader := strings.NewReader("")
	req := httptest.NewRequest("GET", "httpz://localhost:8000", reader)

	rt := x.RoundTripper(http.DefaultTransport)
	_, err := rt.RoundTrip(req)
	assert.Error(t, err)

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, fmt.Sprintf("%v", err), s.Cause.Exceptions[0].Message)
}
