package connections

import (
	"context"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strings"
	"testing"

	"code.justin.tv/foundation/twitchclient"

	"github.com/stretchr/testify/assert"
)

var (
	twitterClient Client
)

func init() {
	serv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		switch {
		case strings.Contains(r.URL.String(), "/v2/user_id/twitter") && r.Method == "DELETE":
			w.WriteHeader(http.StatusNoContent)
		case strings.Contains(r.URL.String(), "/v2/user_id/twitter/auth"):
			data, err := json.Marshal(RedirectPathResp{Path: "redirect"})
			if err != nil {
				panic(err)
			}
			w.Write(data)
		case strings.Contains(r.URL.String(), "/v2/user_id/twitter/callback"):
			w.WriteHeader(http.StatusCreated)
		case strings.Contains(r.URL.String(), "/v2/user_id/twitter") && r.Method == "GET":
			w.WriteHeader(http.StatusOK)
			data, err := json.Marshal(TwitterUser{TwitchID: "twitchID", TwitterID: "twitterID", TwitterScreenName: "twitterScreenName"})
			if err != nil {
				panic(err)
			}
			w.Write(data)
		}
		return
	}))
	client, err := NewClient(twitchclient.ClientConf{Host: serv.URL})
	if err != nil {
		panic(err)
	}
	twitterClient = client
	return
}

func TestAuthTwitter(t *testing.T) {
	res, err := twitterClient.AuthTwitter(context.Background(), "user_id", nil)
	assert.Nil(t, err, "should not return error")
	assert.Equal(t, "redirect", res, "should return redirect path")
}

func TestGetTwitterUser(t *testing.T) {
	res, err := twitterClient.GetTwitterUser(context.Background(), "user_id", nil)
	assert.Nil(t, err, "should not return error")
	assert.Equal(t, "twitchID", res.TwitchID, "should return twitch id")
	assert.Equal(t, "twitterID", res.TwitterID, "should return twitter id")
	assert.Equal(t, "twitterScreenName", res.TwitterScreenName, "should return twitter screen name")
}

func TestPostTwitterUser(t *testing.T) {
	err := twitterClient.CallbackTwitter(context.Background(), "user_id", url.Values{}, nil)
	assert.Nil(t, err, "should not return error")
}

func TestDeleteTwitterUser(t *testing.T) {
	err := twitterClient.DeleteTwitterUser(context.Background(), "user_id", nil)
	assert.Nil(t, err, "should not return error")
}
