package main

import (
	"fmt"
	"log"
	"net"
	"os"
	"strconv"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/afex/hystrix-go/plugins"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/sirupsen/logrus"
	"github.com/stvp/rollbar"

	"code.justin.tv/chat/redicache"
	"code.justin.tv/chat/redis"
	"code.justin.tv/common/config"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/foundation/xray"
	_ "code.justin.tv/foundation/xray/plugins/beanstalk"
	_ "code.justin.tv/foundation/xray/plugins/ec2"
	"code.justin.tv/identity/connections/app/api"
	"code.justin.tv/identity/connections/app/api/controllers"
	blizzardapi "code.justin.tv/identity/connections/app/api/controllers/blizzard"
	twitterapi "code.justin.tv/identity/connections/app/api/controllers/twitter"
	"code.justin.tv/identity/connections/app/backend"
	blizzardbackend "code.justin.tv/identity/connections/app/backend/platforms/blizzard"
	twitterbackend "code.justin.tv/identity/connections/app/backend/platforms/twitter"
	"code.justin.tv/identity/connections/app/clients"
	blizzardclient "code.justin.tv/identity/connections/app/clients/blizzard"
	"code.justin.tv/identity/connections/app/clients/postgres"
	"code.justin.tv/identity/connections/app/clients/steam"
	twitterclient "code.justin.tv/identity/connections/app/clients/twitter"
	"code.justin.tv/identity/connections/app/clients/youtube"
	"code.justin.tv/identity/connections/app/utils/auth"
	"code.justin.tv/identity/connections/app/utils/cryptor"
	"code.justin.tv/identity/connections/app/utils/spade"
	cartman "code.justin.tv/web/cartman/client"
)

func main() {
	// Registered config holds default values that will be overwritten
	// by environment variables. Elastic Beanstalk "Software Configuration"
	// will set environment variables to overwrite these defaults.
	config.Register(map[string]string{
		// App Config
		// These config values are set by jtv/common/config. Do not uncomment.
		//
		// "app":              "connections",
		// "environment":      "development",
		// "statsd-host-port": "statsd.internal.justin.tv:8125",
		// "rollbar-token":    "",

		// Postgres Config
		// Defaults are set to connect to staging sitedb.
		"PG_HOST_MASTER":          "master-sitedb.staging.us-west2.justin.tv",
		"PG_PORT_MASTER":          "6542",
		"PG_HOST_SLAVE":           "master-sitedb.staging.us-west2.justin.tv",
		"PG_PORT_SLAVE":           "6542",
		"PG_USER":                 "change_me",
		"PG_PASSWORD":             "change_me",
		"PG_DATABASE":             "site_justintv_dev",
		"PG_DRIVER_NAME":          "postgres",
		"PG_MAX_OPEN_CONNS":       "500",
		"PG_MAX_IDLE_CONNS":       "250",
		"PG_MAX_QUEUE_SIZE":       "250",
		"PG_CONN_ACQUIRE_TIMEOUT": "1",
		"PG_REQUEST_TIMEOUT":      "2",
		"PG_MAX_CONN_AGE":         "120",

		// RDS Config
		"RDS_HOST_MASTER":          "connections-dev.cekokguegnp4.us-west-2.rds.amazonaws.com",
		"RDS_PORT_MASTER":          "5432",
		"RDS_HOST_SLAVE":           "connections-dev.cekokguegnp4.us-west-2.rds.amazonaws.com",
		"RDS_PORT_SLAVE":           "5432",
		"RDS_DATABASE":             "connections",
		"RDS_USER":                 "connections",
		"RDS_PASSWORD":             "hellomynameisben",
		"RDS_DRIVER_NAME":          "postgres",
		"RDS_MAX_OPEN_CONNS":       "500",
		"RDS_MAX_IDLE_CONNS":       "250",
		"RDS_MAX_QUEUE_SIZE":       "250",
		"RDS_CONN_ACQUIRE_TIMEOUT": "1",
		"RDS_REQUEST_TIMEOUT":      "2",
		"RDS_MAX_CONN_AGE":         "120",

		// SSO Config
		"SSO_HOST_MASTER":          "sso-dev.cpex9kwfj5j5.us-west-2.rds.amazonaws.com",
		"SSO_PORT_MASTER":          "5432",
		"SSO_HOST_SLAVE":           "sso-dev.cpex9kwfj5j5.us-west-2.rds.amazonaws.com",
		"SSO_PORT_SLAVE":           "5432",
		"SSO_DATABASE":             "sso",
		"SSO_USER":                 "sso",
		"SSO_PASSWORD":             "gelatinbeeswaxtea",
		"SSO_DRIVER_NAME":          "postgres",
		"SSO_MAX_OPEN_CONNS":       "500",
		"SSO_MAX_IDLE_CONNS":       "250",
		"SSO_MAX_QUEUE_SIZE":       "250",
		"SSO_CONN_ACQUIRE_TIMEOUT": "1",
		"SSO_REQUEST_TIMEOUT":      "2",
		"SSO_MAX_CONN_AGE":         "120",

		// Twitter Config
		"TWITTER_CONSUMER_KEY":    "change_me",
		"TWITTER_CONSUMER_SECRET": "change_me",
		"TWITTER_ENCRYPTION_KEY":  "This Should Be 32Bytes In Length",
		"TWITTER_CALLBACK_URL":    "https://api.twitch.tv/v5/twitter/callback?id=%s&nonce=%s",

		// Steam Config
		"STEAM_CALLBACK_URL": "http://localhost.twitch.tv:8000/v1/%s/platforms/steam/callback?client_id=%s",
		"STEAM_ROOT_URL":     "http://localhost.twitch.tv:8000",

		// Youtube Config
		"YOUTUBE_CLIENT_ID":     "994605931410-73thh6q8u20ucqn5c5latalk13mf6opu.apps.googleusercontent.com",
		"YOUTUBE_CLIENT_SECRET": "5N_2eerfphFI4W1MTnqS1csm",
		"YOUTUBE_CALLBACK_URL":  "http://api.twitch.tv/v5/youtube/callback",

		// Blizzard Config
		"BLIZZARD_CLIENT_ID":     "kdfwbmk78ct87nfbc72vmuexxefu2eab",
		"BLIZZARD_CLIENT_SECRET": "fxVsf3UTqBD8Zveq4CvFJKpcQR39997s",
		"BLIZZARD_REDIRECT_URL":  "https://api.twitch.tv/v5/blizzard/callback",

		// Redis Config
		"REDIS_HOST":            "localhost",
		"REDIS_PORT":            "6379",
		"REDIS_PASSWORD":        "",
		"REDIS_CONNECT_TIMEOUT": "500",
		"REDIS_READ_TIMEOUT":    "500",
		"REDIS_WRITE_TIMEOUT":   "100",
		"REDIS_MAX_CONNECTIONS": "1000",

		// Cartman Config
		"CARTMAN_HMAC_KEY": "nerf this!",
		"CARTMAN_HOST":     "http://cartman-staging-elb.dev.us-west2.justin.tv",
	})

	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	initLogging()
	initHystrix()

	// Dependencies
	stats := config.Statsd()
	redisClient := initRedis(stats)
	redisCache := initRedicache(redisClient)
	rds := initRDS(stats)
	tracker := initSpade()

	// Platforms
	blizz := initBlizzard(rds, redisClient, redisCache, tracker, stats)
	twtr := initTwitter(rds, redisClient, redisCache, tracker, stats)

	server, err := api.NewServer(
		initBackend(stats, redisCache),
		initSteam(redisClient),
		initYoutube(redisClient),
		blizz,
		twtr,
		initAuthenticator(),
		initCartman(stats),
		tracker,
		stats)
	if err != nil {
		log.Fatal(err)
	}

	if err := xray.Configure(xray.Config{Name: "connections", Sampling: -1.0}); err != nil {
		log.Fatal(err)
	}

	twitchserver.AddDefaultSignalHandlers()
	log.Fatal(twitchserver.ListenAndServe(server, nil))
}

func initRedis(statter statsd.Statter) redis.Redis {
	conf := redicache.Config{
		Host:           config.Resolve("REDIS_HOST"),
		Port:           configResolveInt("REDIS_PORT"),
		Password:       config.Resolve("REDIS_PASSWORD"),
		KeyPrefix:      fmt.Sprintf("connections:%s", config.Resolve("environment")),
		StatsPrefix:    "redis",
		ConnectTimeout: configResolveDuration("REDIS_CONNECT_TIMEOUT"),
		ReadTimeout:    configResolveDuration("REDIS_READ_TIMEOUT"),
		WriteTimeout:   configResolveDuration("REDIS_WRITE_TIMEOUT"),
		MaxConns:       configResolveInt("REDIS_MAX_CONNECTIONS"),
	}

	cache, err := redis.Init{
		Address:   net.JoinHostPort(conf.Host, strconv.Itoa(conf.Port)),
		Password:  conf.Password,
		KeyPrefix: conf.KeyPrefix,

		ConnectTimeout: time.Duration(conf.ConnectTimeout),
		ReadTimeout:    time.Duration(conf.ReadTimeout),
		WriteTimeout:   time.Duration(conf.WriteTimeout),
		MaxConns:       conf.MaxConns,

		XactGroup:   "redis",
		StatsPrefix: conf.StatsPrefix,
		Stats:       statter,
	}.New()

	if err != nil {
		log.Fatalf("failed to initialize redis: %s", err.Error())
	}

	return cache
}

func initRedicache(r redis.Redis) redicache.Cache {
	return redicache.NewFromRedis(r)
}

// Configure SiteDB + Redis Backend
func initBackend(statter statsd.Statter, cache redicache.Cache) backend.ConnectionsBackender {
	pconfMaster := postgres.Config{
		Host:               config.Resolve("PG_HOST_MASTER"),
		Port:               configResolveInt("PG_PORT_MASTER"),
		User:               config.Resolve("PG_USER"),
		Password:           config.Resolve("PG_PASSWORD"),
		Database:           config.Resolve("PG_DATABASE"),
		DriverName:         config.Resolve("PG_DRIVER_NAME"),
		MaxOpenConns:       configResolveInt("PG_MAX_OPEN_CONNS"),
		MaxIdleConns:       configResolveInt("PG_MAX_IDLE_CONNS"),
		MaxQueueSize:       configResolveInt("PG_MAX_QUEUE_SIZE"),
		ConnAcquireTimeout: configResolveInt("PG_CONN_ACQUIRE_TIMEOUT"),
		RequestTimeout:     configResolveInt("PG_REQUEST_TIMEOUT"),
		MaxConnAge:         configResolveInt("PG_MAX_CONN_AGE"),
	}
	pgDbMaster, err := postgres.New(pconfMaster, statter)
	if err != nil {
		log.Fatalf("failed to initialize pg backend: %s", err.Error())
	}

	pconfSlave := postgres.Config{
		Host:               config.Resolve("PG_HOST_SLAVE"),
		Port:               configResolveInt("PG_PORT_SLAVE"),
		User:               config.Resolve("PG_USER"),
		Password:           config.Resolve("PG_PASSWORD"),
		Database:           config.Resolve("PG_DATABASE"),
		DriverName:         config.Resolve("PG_DRIVER_NAME"),
		MaxOpenConns:       configResolveInt("PG_MAX_OPEN_CONNS"),
		MaxIdleConns:       configResolveInt("PG_MAX_IDLE_CONNS"),
		MaxQueueSize:       configResolveInt("PG_MAX_QUEUE_SIZE"),
		ConnAcquireTimeout: configResolveInt("PG_CONN_ACQUIRE_TIMEOUT"),
		RequestTimeout:     configResolveInt("PG_REQUEST_TIMEOUT"),
		MaxConnAge:         configResolveInt("PG_MAX_CONN_AGE"),
	}
	pgDbSlave, err := postgres.New(pconfSlave, statter)
	if err != nil {
		log.Fatalf("failed to initialize pg backend: %s", err.Error())
	}

	rconfMaster := postgres.Config{
		Host:               config.Resolve("RDS_HOST_MASTER"),
		Port:               configResolveInt("RDS_PORT_MASTER"),
		User:               config.Resolve("RDS_USER"),
		Password:           config.Resolve("RDS_PASSWORD"),
		Database:           config.Resolve("RDS_DATABASE"),
		DriverName:         config.Resolve("RDS_DRIVER_NAME"),
		MaxOpenConns:       configResolveInt("RDS_MAX_OPEN_CONNS"),
		MaxIdleConns:       configResolveInt("RDS_MAX_IDLE_CONNS"),
		MaxQueueSize:       configResolveInt("RDS_MAX_QUEUE_SIZE"),
		ConnAcquireTimeout: configResolveInt("RDS_CONN_ACQUIRE_TIMEOUT"),
		RequestTimeout:     configResolveInt("RDS_REQUEST_TIMEOUT"),
		MaxConnAge:         configResolveInt("RDS_MAX_CONN_AGE"),
	}

	rdsDbMaster, err := postgres.New(rconfMaster, statter)
	if err != nil {
		log.Fatalf("failed to initialize rds backend: %s", err.Error())
	}

	rconfSlave := postgres.Config{
		Host:               config.Resolve("RDS_HOST_SLAVE"),
		Port:               configResolveInt("RDS_PORT_SLAVE"),
		User:               config.Resolve("RDS_USER"),
		Password:           config.Resolve("RDS_PASSWORD"),
		Database:           config.Resolve("RDS_DATABASE"),
		DriverName:         config.Resolve("RDS_DRIVER_NAME"),
		MaxOpenConns:       configResolveInt("RDS_MAX_OPEN_CONNS"),
		MaxIdleConns:       configResolveInt("RDS_MAX_IDLE_CONNS"),
		MaxQueueSize:       configResolveInt("RDS_MAX_QUEUE_SIZE"),
		ConnAcquireTimeout: configResolveInt("RDS_CONN_ACQUIRE_TIMEOUT"),
		RequestTimeout:     configResolveInt("RDS_REQUEST_TIMEOUT"),
		MaxConnAge:         configResolveInt("RDS_MAX_CONN_AGE"),
	}

	rdsDbSlave, err := postgres.New(rconfSlave, statter)
	if err != nil {
		log.Fatalf("failed to initialize rds backend: %s", err.Error())
	}

	ssoConfMaster := postgres.Config{
		Host:               config.Resolve("SSO_HOST_MASTER"),
		Port:               configResolveInt("SSO_PORT_MASTER"),
		User:               config.Resolve("SSO_USER"),
		Password:           config.Resolve("SSO_PASSWORD"),
		Database:           config.Resolve("SSO_DATABASE"),
		DriverName:         config.Resolve("SSO_DRIVER_NAME"),
		MaxOpenConns:       configResolveInt("SSO_MAX_OPEN_CONNS"),
		MaxIdleConns:       configResolveInt("SSO_MAX_IDLE_CONNS"),
		MaxQueueSize:       configResolveInt("SSO_MAX_QUEUE_SIZE"),
		ConnAcquireTimeout: configResolveInt("SSO_CONN_ACQUIRE_TIMEOUT"),
		RequestTimeout:     configResolveInt("SSO_REQUEST_TIMEOUT"),
		MaxConnAge:         configResolveInt("SSO_MAX_CONN_AGE"),
	}

	ssoDbMaster, err := postgres.New(ssoConfMaster, statter)
	if err != nil {
		log.Fatalf("failed to initialize sso backend: %s", err.Error())
	}

	ssoConfSlave := postgres.Config{
		Host:               config.Resolve("SSO_HOST_SLAVE"),
		Port:               configResolveInt("SSO_PORT_SLAVE"),
		User:               config.Resolve("SSO_USER"),
		Password:           config.Resolve("SSO_PASSWORD"),
		Database:           config.Resolve("SSO_DATABASE"),
		DriverName:         config.Resolve("SSO_DRIVER_NAME"),
		MaxOpenConns:       configResolveInt("SSO_MAX_OPEN_CONNS"),
		MaxIdleConns:       configResolveInt("SSO_MAX_IDLE_CONNS"),
		MaxQueueSize:       configResolveInt("SSO_MAX_QUEUE_SIZE"),
		ConnAcquireTimeout: configResolveInt("SSO_CONN_ACQUIRE_TIMEOUT"),
		RequestTimeout:     configResolveInt("SSO_REQUEST_TIMEOUT"),
		MaxConnAge:         configResolveInt("SSO_MAX_CONN_AGE"),
	}

	ssoDbSlave, err := postgres.New(ssoConfSlave, statter)
	if err != nil {
		log.Fatalf("failed to initialize sso backend: %s", err.Error())
	}

	return backend.NewPGBackend(pgDbMaster, pgDbSlave, rdsDbMaster, rdsDbSlave, ssoDbMaster, ssoDbSlave, statter, cache)
}

func initSitedb(stats statsd.Statter) *postgres.DB {
	dbs, err := newPostgresDB("PG", stats)
	if err != nil {
		log.Fatalf("unable to initialize pg backend: %s", err.Error())
	}
	return dbs
}

func initRDS(stats statsd.Statter) *postgres.DB {
	dbs, err := newPostgresDB("RDS", stats)
	if err != nil {
		log.Fatalf("unable to initialize rds backend: %s", err.Error())
	}
	return dbs
}

const (
	masterSuffix = "MASTER"
	slaveSuffix  = "SLAVE"
)

func newPostgresDB(prefix string, stats statsd.Statter) (*postgres.DB, error) {
	master := postgresConfig(prefix, masterSuffix)
	slave := postgresConfig(prefix, slaveSuffix)
	return postgres.NewDBs(master, slave, stats)
}

func postgresConfig(prefix, suffix string) postgres.Config {
	conf := basePostgresConfig(prefix)
	conf.Host = config.Resolve(prefix + "_HOST_" + suffix)
	conf.Port = configResolveInt(prefix + "_PORT_" + suffix)
	conf.User = config.Resolve(prefix + "_USER")
	conf.Password = config.Resolve(prefix + "_PASSWORD")
	conf.Database = config.Resolve(prefix + "_DATABASE")
	return conf
}

func basePostgresConfig(prefix string) postgres.Config {
	return postgres.Config{
		DriverName:         config.Resolve(prefix + "_DRIVER_NAME"),
		MaxOpenConns:       configResolveInt(prefix + "_MAX_OPEN_CONNS"),
		MaxIdleConns:       configResolveInt(prefix + "_MAX_IDLE_CONNS"),
		MaxQueueSize:       configResolveInt(prefix + "_MAX_QUEUE_SIZE"),
		ConnAcquireTimeout: configResolveInt(prefix + "_CONN_ACQUIRE_TIMEOUT"),
		RequestTimeout:     configResolveInt(prefix + "_REQUEST_TIMEOUT"),
		MaxConnAge:         configResolveInt(prefix + "_MAX_CONN_AGE"),
	}
}

func initCartman(stats statsd.Statter) cartman.Client {
	client, err := cartman.NewClient(twitchclient.ClientConf{
		Host:           config.Resolve("CARTMAN_HOST"),
		Stats:          stats,
		TimingXactName: "cartman",
	})
	if err != nil {
		log.Fatalf("unable to initialize cartman client: %q", err)
	}
	return client
}

func initSteam(r redis.Redis) clients.Client {
	client, err := steam.NewClient(
		config.Resolve("STEAM_CALLBACK_URL"),
		config.Resolve("STEAM_ROOT_URL"),
		r,
	)
	if err != nil {
		log.Fatalf("failed to initialize steam client: %s", err.Error())
	}
	return client
}

func initYoutube(r redis.Redis) clients.Client {
	client, err := youtube.NewClient(
		youtube.Config{
			ClientID:     config.Resolve("YOUTUBE_CLIENT_ID"),
			ClientSecret: config.Resolve("YOUTUBE_CLIENT_SECRET"),
			RedirectURL:  config.Resolve("YOUTUBE_CALLBACK_URL"),
		},
		r,
	)
	if err != nil {
		log.Fatalf("failed to initialize youtube client: %s", err.Error())
	}
	return client
}

func initBlizzard(db *postgres.DB, store redis.Redis, cache redicache.Cache, tracker spade.Client, stats statsd.Statter) controllers.Controller {
	backend := blizzardbackend.NewBlizzardBackend(db, cache, stats)
	client := blizzardclient.NewClient(
		config.Resolve("BLIZZARD_CLIENT_ID"),
		config.Resolve("BLIZZARD_CLIENT_SECRET"),
		config.Resolve("BLIZZARD_REDIRECT_URL"),
		store)
	controller := blizzardapi.NewBlizzardController(backend, client, tracker)
	return controller
}

func initTwitter(db *postgres.DB, store redis.Redis, cache redicache.Cache, tracker spade.Client, stats statsd.Statter) controllers.Controller {
	crypt, err := cryptor.NewCryptor(config.Resolve("TWITTER_ENCRYPTION_KEY"))
	if err != nil {
		log.Fatalf("unable to initialize twitter cryptor: %q", err)
	}
	backend := twitterbackend.NewTwitterBackend(db, cache, crypt, stats)
	client := twitterclient.NewClient(
		config.Resolve("TWITTER_CONSUMER_KEY"),
		config.Resolve("TWITTER_CONSUMER_SECRET"),
		config.Resolve("TWITTER_CALLBACK_URL"),
		store)
	controller := twitterapi.NewTwitterController(backend, client, tracker)
	return controller
}

func initSpade() spade.Client {
	client, err := spade.NewClient()
	if err != nil {
		log.Fatalf("unable to initialize spade client: %q", err)
	}
	return client
}

func initAuthenticator() auth.Authenticator {
	auther, err := auth.NewAuthenticator(config.Resolve("CARTMAN_HMAC_KEY"))
	if err != nil {
		log.Fatalf("failed to initialize authenticator: %s", err.Error())
	}
	return auther
}

func initLogging() {
	rollbar.Token = config.RollbarToken()
	rollbar.Environment = config.Environment()
	logrus.SetFormatter(&logrus.TextFormatter{FullTimestamp: true, TimestampFormat: time.RFC1123Z})
}

func initHystrix() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"spade": hystrix.CommandConfig{
			Timeout:                1000,
			MaxConcurrentRequests:  1000,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		// The redis command is given a larger timeout and max pool of concurrent
		// requests due to it calling an encapsulated hystrix command for "sitedb".
		"redis": hystrix.CommandConfig{
			Timeout:                2000,
			MaxConcurrentRequests:  1000,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"sitedb_query": hystrix.CommandConfig{
			Timeout:                500,
			MaxConcurrentRequests:  300,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"sitedb_exec": hystrix.CommandConfig{
			Timeout:                500,
			MaxConcurrentRequests:  300,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"rds_query": hystrix.CommandConfig{
			Timeout:                300,
			MaxConcurrentRequests:  300,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"rds_exec": hystrix.CommandConfig{
			Timeout:                300,
			MaxConcurrentRequests:  300,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"getTwitterAccessToken": hystrix.CommandConfig{
			Timeout:                500,
			MaxConcurrentRequests:  100,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"exchangeYoutubeCode": hystrix.CommandConfig{
			Timeout:                2000,
			MaxConcurrentRequests:  100,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"getYoutubeID": hystrix.CommandConfig{
			Timeout:                2000,
			MaxConcurrentRequests:  100,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"refreshYoutubeToken": hystrix.CommandConfig{
			Timeout:                2000,
			MaxConcurrentRequests:  100,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"blizzard": hystrix.CommandConfig{
			Timeout:                2000,
			MaxConcurrentRequests:  300,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"blizzard_oauth": hystrix.CommandConfig{
			Timeout:                4000,
			MaxConcurrentRequests:  100,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"cartman": hystrix.CommandConfig{
			Timeout:                1000,
			MaxConcurrentRequests:  500,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"twitter": hystrix.CommandConfig{
			Timeout:                500,
			MaxConcurrentRequests:  100,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
		"twitter_oauth": hystrix.CommandConfig{
			Timeout:                500,
			MaxConcurrentRequests:  100,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000,
			ErrorPercentThreshold:  50,
		},
	})

	host, err := os.Hostname()
	if err != nil {
		host = "unknown"
	}
	c, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
		StatsdAddr: config.StatsdHostPort(),
		Prefix:     fmt.Sprintf("%s.%s.%s.hystrix", "connections", config.Environment(), host),
	})
	if err != nil {
		log.Fatalf("failed to initailize hystrix statsd client: %s", err)
		return
	}
	metricCollector.Registry.Register(c.NewStatsdCollector)
}

func configResolveInt(field string) int {
	i, err := strconv.Atoi(config.Resolve(field))
	if err != nil {
		log.Fatal(fmt.Sprintf("config parameter '%s' is not valid: ", field), err)
	}
	return i
}

func configResolveDuration(field string) time.Duration {
	i := configResolveInt(field)
	return time.Duration(i) * time.Millisecond
}
