package associations

import (
	"fmt"
	"strconv"
	"time"
)

// Global schema object that is the authority on what parameters are allowed for the cohesion API
var SchemaManager *Schema

// AssocKind describes an association kind, its inverse (if it has one), and which
// EntityKinds either side of the association has to be
type AssocKind struct {
	KindStr    string `json:"kind"`
	InverseStr string `json:"inverse"`
	E1KindStr  string `json:"e1Kind"`
	E2KindStr  string `json:"e2Kind"`
	ShardFrom  bool   `json:"shardFrom"`
}

// EntityKind describes an entity kind, like 'user'
type EntityKind struct {
	KindStr string `json:"kind"`
}

func (e *EntityKind) String() string {
	return e.KindStr
}

func (a *AssocKind) String() string {
	return a.KindStr
}

// Kind returns the association kind string
func (a *AssocKind) Kind() string {
	return a.String()
}

//ShardFrom returns whether migration partition sharding should shard on the from entity in this association
//(if false, shard on the to entity)
func (a *AssocKind) ShardOnFromEntity() bool {
	return a.ShardFrom
}

// Inverse returns the inverse of this association kind, if it's defined
func (a *AssocKind) Inverse() (AssocKind, bool) {
	kind, ok := SchemaManager.AssocKind(a.InverseStr)
	if !ok {
		kind = SchemaManager.UnknownAssocKind
	}
	return kind, ok
}

// Schema holds the configuration schema file in memory. This is all AssocKinds and EntityKinds
type Schema struct {
	AssocSchema       map[string]AssocKind
	EntitySchema      map[string]EntityKind
	UnknownAssocKind  AssocKind
	UnknownEntityKind EntityKind
	EmptyEntity       Entity
}

// AssocKind returns the AssocKind that corresponds to the given string, if it exists
func (s *Schema) AssocKind(kind string) (AssocKind, bool) {
	assocKind, ok := s.AssocSchema[kind]
	if !ok {
		assocKind = s.UnknownAssocKind
	}
	return assocKind, ok
}

// EntityKind returns the EntityKind that corresponds to the given string, if it exists
func (s *Schema) EntityKind(kind string) (EntityKind, bool) {
	entityKind, ok := s.EntitySchema[kind]
	if !ok {
		entityKind = s.UnknownEntityKind
	}
	return entityKind, ok
}

// Validate validates that an association with the given two entity kinds and association kind
// is valid according to the loaded schema
func (s *Schema) Validate(e1Kind, assocKind, e2Kind string) error {
	validatedE1Kind, ok := s.EntityKind(e1Kind)
	if !ok {
		return fmt.Errorf("Invalid e1 kind: %s", e1Kind)
	}

	validatedE2Kind, ok := s.EntityKind(e2Kind)
	if !ok {
		return fmt.Errorf("Invalid e2 kind: %s", e2Kind)
	}

	validatedAssocKind, ok := s.AssocKind(assocKind)
	if !ok {
		return fmt.Errorf("Invalid assoc kind: %s", assocKind)
	}

	assocE1Kind, _ := s.EntityKind(validatedAssocKind.E1KindStr)
	if validatedE1Kind != assocE1Kind {
		return fmt.Errorf(
			"Invalid e1 kind: %s for association %s, expected %s",
			validatedE1Kind.String(),
			validatedAssocKind.Kind(),
			validatedAssocKind.E1KindStr,
		)
	}

	assocE2Kind, _ := s.EntityKind(validatedAssocKind.E2KindStr)
	if validatedE2Kind != assocE2Kind {
		return fmt.Errorf(
			"Invalid e2 kind: %s for association %s, expected %s",
			validatedE2Kind.String(),
			validatedAssocKind.Kind(),
			validatedAssocKind.E2KindStr,
		)
	}

	return nil
}

// NewSchemaManager constructs a SchemaManager
func NewSchemaManager(schema string) (*Schema, error) {
	assocSchema, entitySchema, err := ReadSchema(schema + ".json")

	if err != nil {
		return nil, err
	}

	schemaManager := Schema{
		AssocSchema:  map[string]AssocKind{},
		EntitySchema: map[string]EntityKind{},
	}

	for assocKind, definition := range assocSchema {
		inverse := "unknown"
		if _, ok := definition["inverse"]; ok {
			inverse = definition["inverse"]
		}

		shardFrom, err := strconv.ParseBool(definition["shardFrom"])
		if err != nil {
			fmt.Println(assocSchema)
			return nil, err
		}
		schemaManager.AssocSchema[assocKind] = AssocKind{
			KindStr:    assocKind,
			InverseStr: inverse,
			E1KindStr:  definition["e1Kind"],
			E2KindStr:  definition["e2Kind"],
			ShardFrom:  shardFrom,
		}
	}

	for entityKind := range entitySchema {
		schemaManager.EntitySchema[entityKind] = EntityKind{
			KindStr: entityKind,
		}
	}

	schemaManager.UnknownAssocKind = AssocKind{KindStr: "unknown"}
	schemaManager.UnknownEntityKind = EntityKind{KindStr: "unknown"}
	schemaManager.EmptyEntity = Entity{Kind: schemaManager.UnknownEntityKind}

	return &schemaManager, nil
}

// DataBag is a map of strings to interfaces, used to pass data about
// associations to and from the client
type DataBag map[string]interface{}

// Equals returns true if every value in d also exists in 'in', and
// those values are equivalent. Not this isn't pure map equality.
func (d DataBag) Equals(in Comparable) bool {
	if b, ok := in.(DataBag); ok {
		for k, v := range d {
			if b[k] != v {
				return false
			}
		}
		return true
	}
	return false
}

// AssocResponse is a utility struct allowing the mapping of association
// meta data to the relevent entity passed back from the backend
type AssocResponse struct {
	E      *Entity
	T      time.Time
	D      DataBag
	Cursor string
}

// Equals returns true if a is equal to in. We disregard time here.
func (a *AssocResponse) Equals(in Comparable) bool {
	if b, ok := in.(*AssocResponse); ok {
		if a.E.Equals(b.E) && a.D.Equals(b.D) {
			return true
		}
	}
	return false
}

type AssocResponseWithMeta struct {
	Kind string
	A    *AssocResponse
}

// Equals returns true if a is equal to in. We disregard time here.
func (a *AssocResponseWithMeta) Equals(in Comparable) bool {
	if b, ok := in.(*AssocResponseWithMeta); ok {
		if a.Kind == b.Kind && a.A.Equals(b.A) {
			return true
		}
	}
	return false
}

// Association represents represents a request for an association.
type Association struct {
	E1   Entity    `json:"e1"`
	E2   Entity    `json:"e2"`
	Kind AssocKind `json:"kind"` // TODO: handle rendering to json
	D    DataBag   `json:"databag"`
}

func (a Association) String() string {
	return fmt.Sprintf("<%s, %s, %s>", a.Kind.String(), a.E1, a.E2)
}

// NewAssoc takes information about an entity, validates it and returns an
// Association containing it
func NewAssoc(e1, e2 Entity, kind string) (Association, error) {
	var assoc Association
	if err := e1.Validate(); err != nil {
		return assoc, err
	}
	assoc.E1 = e1

	if err := e2.Validate(); err != nil {
		return assoc, err
	}
	assoc.E2 = e2

	var ok bool
	if assoc.Kind, ok = SchemaManager.AssocKind(kind); !ok {
		return assoc, ErrInvalidKind{"Invalid Kind specified"}
	}

	if err := assoc.validate(); err != nil {
		return assoc, err
	}

	return assoc, nil
}

// Validate checks the association kind against the schema and
// ensures the associated entities are the correct kind.
func (a *Association) validate() error {
	if a.Kind == SchemaManager.UnknownAssocKind {
		return ErrInvalidKind{"Invalid Kind specified"}
	}

	e1Kind, ok := SchemaManager.EntityKind(a.Kind.E1KindStr)
	if !ok {
		return fmt.Errorf("Invalid entity1Kind kind: %s", e1Kind)
	}

	if a.E1.Kind != e1Kind {
		return ErrInvalidKind{fmt.Sprintf("Invalid entity 1 kind %v for association kind %v", a.E1.Kind, a.Kind)}
	}

	if a.E2.Empty() {
		return nil
	}

	e2Kind, ok := SchemaManager.EntityKind(a.Kind.E2KindStr)
	if !ok {
		return fmt.Errorf("Invalid entity2Kind kind: %s", e2Kind)
	}

	if a.E2.Kind != e2Kind {
		return ErrInvalidKind{fmt.Sprintf("Invalid entity 2 kind %v for association kind %v", a.E2.Kind, a.Kind)}
	}

	if a.E1.ID == a.E2.ID && a.E1.Kind == a.E2.Kind {
		return ErrInvalidID{"Entities cannot be identical"}
	}

	return nil
}
