package associations

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"
	"strings"
)

var schema schemaJSON
var validEntities map[string]bool

type schemaJSON struct {
	AssocSchema  map[string]map[string]string `json:"AssocSchema"`
	EntitySchema []string                     `json:"EntitySchema"`
}

func validateInverses() []string {
	inverses := map[string]*string{}
	errors := []string{}

	for kind, definition := range schema.AssocSchema {
		if inverse, ok := definition["inverse"]; ok {
			inverses[kind] = &inverse
		} else {
			inverses[kind] = nil
		}
	}

	for kind, inverse := range inverses {
		if inverse == nil {
			continue
		}

		if expectedKind, ok := inverses[*inverse]; ok {
			if expectedKind == nil {
				errors = append(errors, fmt.Sprintf("\"%s\" missing expected inverse of \"%s\"", *inverse, kind))
			} else if kind != *expectedKind {
				errors = append(errors, fmt.Sprintf("Inverse definitions for \"%s\" and \"%s\" do not agree", kind, *inverse))
			}
		} else {
			errors = append(errors, fmt.Sprintf("AssocSchema \"%s\" expected but not defined", *inverse))
		}
	}

	return errors
}

func validateEntitySchemas() []string {
	errors := []string{}

	for kind, definition := range schema.AssocSchema {
		if EntitySchema, ok := definition["e1Kind"]; ok {
			if _, ok := validEntities[EntitySchema]; !ok {
				errors = append(errors, fmt.Sprintf("AssocSchema \"%s\" has an invalid e1Kind: \"%s\"", kind, EntitySchema))
			}
		} else {
			errors = append(errors, fmt.Sprintf("AssocSchema \"%s\" missing key \"e1Kind\"", kind))
		}

		if EntitySchema, ok := definition["e2Kind"]; ok {
			if _, ok := validEntities[EntitySchema]; !ok {
				errors = append(errors, fmt.Sprintf("AssocSchema \"%s\" has an invalid e2Kind: \"%s\"", kind, EntitySchema))
			}
		} else {
			errors = append(errors, fmt.Sprintf("AssocSchema \"%s\" missing key \"e2Kind\"", kind))
		}
	}

	return errors
}

func generateBasePaths() []string {
	paths := []string{}

	currentWd, err := os.Getwd()
	if err == nil {
		paths = append(paths, currentWd+"/schema/")
	}

	if os.Getenv("COHESION_SCHEMA_FOLDER") != "" {
		paths = append(paths, os.Getenv("COHESION_SCHEMA_FOLDER"))
	}

	if os.Getenv("GOPATH") != "" {
		goPaths := strings.Split(os.Getenv("GOPATH"), ":")
		for _, path := range goPaths {
			paths = append(paths, path+"/src/code.justin.tv/web/cohesion/schema/")
		}
	}

	return paths
}

// ListSchemaFiles returns the names of the schema files that were found
func ListSchemaFiles() []string {
	basePaths := generateBasePaths()
	fileNames := []string{}
	seenFiles := map[string]bool{}

	for _, basePath := range basePaths {
		files, err := ioutil.ReadDir(basePath)
		if err != nil {
			continue
		}
		for _, f := range files {
			if _, ok := seenFiles[f.Name()]; !ok && filepath.Ext(f.Name()) == ".json" {
				seenFiles[f.Name()] = true
				fileNames = append(fileNames, f.Name())
			}
		}
	}

	return fileNames
}

// ReadSchema loads the schema in the specified filename.
// Returns the possible association kinds and entities that the schema supports.
func ReadSchema(fileName string) (map[string]map[string]string, map[string]bool, error) {
	var file *os.File
	var err error

	basePaths := generateBasePaths()

	for _, basePath := range basePaths {
		if _, err = os.Stat(path.Join(basePath, fileName)); err == nil {
			file, err = os.Open(path.Join(basePath, fileName))
			if err == nil {
				break
			}
		}
	}

	if err == nil {
		defer func() { _ = file.Close() }()
	} else {
		quotedPaths := make([]string, len(basePaths))
		copy(quotedPaths, basePaths)

		for i, path := range quotedPaths {
			quotedPaths[i] = `'` + path + `'`
		}
		return nil, nil, fmt.Errorf(
			"Could not find %s in any of the following paths: %s",
			fileName,
			strings.Join(quotedPaths, ", "),
		)
	}

	bytes, err := ioutil.ReadAll(file)
	if err != nil {
		return nil, nil, err
	}

	schema = schemaJSON{}

	err = json.Unmarshal(bytes, &schema)
	if err != nil {
		return nil, nil, err
	}

	if len(schema.AssocSchema) == 0 {
		return nil, nil, errors.New("Error, expected \"AssocSchema\" as top level key in schema")
	}

	if len(schema.EntitySchema) == 0 {
		return nil, nil, errors.New("Error, expected \"EntitySchema\" as top level key in schema")
	}

	validEntities = map[string]bool{}
	schemaErrors := validateInverses()

	for _, EntitySchema := range schema.EntitySchema {
		validEntities[EntitySchema] = true
	}

	schemaErrors = append(schemaErrors, validateEntitySchemas()...)

	if len(schemaErrors) != 0 {
		for i := range schemaErrors {
			schemaErrors[i] = "* " + schemaErrors[i]
		}
		return nil, nil, errors.New("\n" + strings.Join(schemaErrors, "\n"))
	}

	return schema.AssocSchema, validEntities, nil
}
