package cohesion

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"golang.org/x/net/context"

	"code.justin.tv/web/cohesion/api/responses/v1"
	"code.justin.tv/web/cohesion/associations"
	. "github.com/smartystreets/goconvey/convey"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/jixwanwang/apiutils"
)

const (
	testHost = "localhost:8000"
)

func expectedHandler(f func(w http.ResponseWriter, r *http.Request)) (*httptest.Server, *bool) {
	executed := false
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		f(w, r)
		executed = true
	})), &executed
}

func TestRepositoryHeader(t *testing.T) {
	Convey("when the calling repository is configured", t, func() {
		Convey("the Twitch-Repository is added to requests", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Header.Get("Twitch-Repository"), ShouldEqual, "code.justin.tv/chat/tmi")
			})
			defer server.Close()
			stats, _ := statsd.NewNoopClient()

			httpClient := DefaultHTTPClient(stats, "test")
			client, err := NewClient(server.URL, httpClient, "test", SetRepository("chat/tmi"))
			So(err, ShouldBeNil)

			_, err = client.FetchAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"})
			So(*executed, ShouldBeTrue)
		})
	})
}

func TestCreateAssoc(t *testing.T) {
	Convey("CreateAssoc", t, func() {
		stats, _ := statsd.NewNoopClient()
		httpClient := DefaultHTTPClient(stats, "test")

		Convey("makes the expected request", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Method, ShouldEqual, "PUT")
				c.So(r.URL.String(), ShouldEqual, "/v1/associations/user/1/follows/user/2")
			})
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.CreateAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"}, map[string]interface{}{})
			So(*executed, ShouldBeTrue)
		})

		Convey("returns nil if the status code is http.StatusCreated", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusCreated)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.CreateAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"}, map[string]interface{}{})
			So(err, ShouldBeNil)
		})

		Convey("returns AssociationExists error if the status code is StatusUnprocessableEntity", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(apiutils.StatusUnprocessableEntity)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.CreateAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"}, map[string]interface{}{})
			So(err, ShouldHaveSameTypeAs, associations.ErrAssociationExists{})
		})
	})
}

func TestDeleteAssoc(t *testing.T) {
	Convey("DeleteAssoc", t, func() {
		stats, _ := statsd.NewNoopClient()
		httpClient := DefaultHTTPClient(stats, "test")

		Convey("makes the expected request", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Method, ShouldEqual, "DELETE")
				c.So(r.URL.String(), ShouldEqual, "/v1/associations/user/1/follows/user/2")
			})
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.DeleteAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"})
			So(*executed, ShouldBeTrue)
		})

		Convey("returns nil if the status code is http.StatusNoContent", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNoContent)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.DeleteAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"})
			So(err, ShouldBeNil)
		})

		Convey("returns ErrNotFound if the status code is http.StatusNotFound", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNotFound)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.DeleteAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"})
			So(err, ShouldHaveSameTypeAs, associations.ErrNotFound{})
		})
	})
}

func TestBulkDeleteAssoc(t *testing.T) {
	Convey("BulkDeleteAssoc", t, func() {
		stats, _ := statsd.NewNoopClient()
		httpClient := DefaultHTTPClient(stats, "test")

		Convey("makes the expected request", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Method, ShouldEqual, "DELETE")
				c.So(r.URL.String(), ShouldEqual, "/v1/associations/user/1/follows/user")
			})
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.BulkDeleteAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", "user")
			So(*executed, ShouldBeTrue)
		})

		Convey("returns nil if the status code is http.StatusNoContent", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNoContent)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.BulkDeleteAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", "user")
			So(err, ShouldBeNil)
		})

		Convey("returns ErrNotFound if the status code is http.StatusNotFound", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNotFound)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.BulkDeleteAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", "user")
			So(err, ShouldHaveSameTypeAs, associations.ErrNotFound{})
		})
	})
}

func TestUpdateAssoc(t *testing.T) {
	Convey("UpdateAssoc", t, func() {
		stats, _ := statsd.NewNoopClient()
		httpClient := DefaultHTTPClient(stats, "test")
		databag := map[string]interface{}{
			"new_assoc_kind": "friends_with",
		}

		Convey("makes the expected request", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Method, ShouldEqual, "POST")
				c.So(r.URL.String(), ShouldEqual, "/v1/associations/user/1/friend_request/user/2")
			})
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.UpdateAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "friend_request", Entity{ID: "2", Kind: "user"}, "", databag)
			So(*executed, ShouldBeTrue)
		})

		Convey("returns nil if the status code is http.StatusNoContent", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNoContent)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.UpdateAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "friend_request", Entity{ID: "2", Kind: "user"}, "", databag)
			So(err, ShouldBeNil)
		})

		Convey("returns ErrNotFound if the status code is http.StatusNotFound", func() {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNotFound)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			err = client.UpdateAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "friend_request", Entity{ID: "2", Kind: "user"}, "", databag)
			So(err, ShouldHaveSameTypeAs, associations.ErrNotFound{})
		})
	})
}

func TestFetchAssoc(t *testing.T) {
	Convey("CreateAssoc", t, func() {
		stats, _ := statsd.NewNoopClient()
		httpClient := DefaultHTTPClient(stats, "test")

		Convey("makes the expected reuest", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Method, ShouldEqual, "GET")
				c.So(r.URL.String(), ShouldEqual, "/v1/associations/user/1/follows/user/2")
			})
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			_, err = client.FetchAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"})
			So(*executed, ShouldBeTrue)
		})

		Convey("parses the response body into an association", func(c C) {
			body := `{"from":{"id":"1","kind":"user"},"kind":"follows","to":[{"target":{"id":"2","kind":"user"},"data":{},"time":"2013-06-21T23:42:48.938547Z"}]}`
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				_, err := w.Write([]byte(body))
				c.So(err, ShouldBeNil)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			assocs, err := client.FetchAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", Entity{ID: "2", Kind: "user"})
			So(err, ShouldBeNil)

			expected := &v1.Response{}
			_ = json.Unmarshal([]byte(body), expected)

			So(expected, ShouldResemble, assocs)
		})
	})
}

func TestListAssoc(t *testing.T) {
	Convey("ListAssoc", t, func() {
		stats, _ := statsd.NewNoopClient()
		httpClient := DefaultHTTPClient(stats, "test")

		Convey("makes the expected request", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Method, ShouldEqual, "GET")
				c.So(r.URL.String(), ShouldEqual, "/v1/associations/user/1/follows/user?limit=100&offset=0&sort=asc")
			})
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			_, err = client.ListAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", "user")
			So(*executed, ShouldBeTrue)
		})

		Convey("parses the response body into a list of associations", func(c C) {
			body := `{"from":{"id":"1","kind":"user"},"kind":"follows","to":[{"target":{"id":"2","kind":"user"},"data":{},"time":"2013-06-21T23:41:29.061639Z"},{"target":{"id":"3","kind":"user"},"data":{},"time":"2013-06-21T23:42:48.938547Z"}]}`
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				_, err := w.Write([]byte(body))
				c.So(err, ShouldBeNil)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			assocs, err := client.ListAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", "user")
			So(err, ShouldBeNil)

			expected := &v1.Response{}
			_ = json.Unmarshal([]byte(body), expected)

			So(expected, ShouldResemble, assocs)
		})
	})
}

func TestCountAssoc(t *testing.T) {
	Convey("CountAssoc", t, func() {
		stats, _ := statsd.NewNoopClient()
		httpClient := DefaultHTTPClient(stats, "test")

		Convey("makes the expected request", func(c C) {
			server, executed := expectedHandler(func(w http.ResponseWriter, r *http.Request) {
				c.So(r.Method, ShouldEqual, "GET")
				c.So(r.URL.String(), ShouldEqual, "/v1/associations/user/1/follows/user/count")
			})
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			_, err = client.CountAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", "user")
			So(*executed, ShouldBeTrue)
		})

		Convey("parses the response body into an int", func(c C) {
			body := `{"count":5}`
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				_, err := w.Write([]byte(body))
				c.So(err, ShouldBeNil)
			}))
			defer server.Close()

			client, err := NewClient(server.URL, httpClient, "test")
			So(err, ShouldBeNil)

			count, err := client.CountAssoc(context.Background(), Entity{ID: "1", Kind: "user"}, "follows", "user")
			So(err, ShouldBeNil)

			So(count, ShouldEqual, 5)
		})
	})
}
