package v2

import (
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/web/cohesion/rpc"

	. "github.com/smartystreets/goconvey/convey"
)

func TestNewListResponse(t *testing.T) {
	Convey("When creating a new ListResponse", t, func() {
		now := time.Now()
		bytes1, _ := json.Marshal(map[string]interface{}{})
		bytes2, _ := json.Marshal(map[string]interface{}{
			"thing1": "thing2",
		})
		resp := &rpc.ListResponseV2{
			From: &rpc.Entity{Name: "123", Kind: "user"},
			Kind: "follows",
			To: []*rpc.AssocResponse{
				&rpc.AssocResponse{
					Target: &rpc.Entity{Name: "456", Kind: "user"},
					Data:   bytes2,
					Time:   &rpc.Timestamp{Seconds: now.Unix(), Nanos: int64(now.Nanosecond())},
				},
				&rpc.AssocResponse{
					Target: &rpc.Entity{Name: "789", Kind: "user"},
					Data:   bytes1,
					Time:   &rpc.Timestamp{Seconds: now.Unix(), Nanos: int64(now.Nanosecond())},
				},
			},
		}

		result := newListResponse(resp)

		So(result.Entity, ShouldResemble, Entity{ID: "123", Kind: "user"})
		So(result.Kind, ShouldEqual, resp.Kind)
		So(len(result.Associations), ShouldEqual, len(resp.To))
		So(result.Associations[0].Time, ShouldResemble, now)
		So(result.Associations[0].DataBag, ShouldResemble, map[string]interface{}{
			"thing1": "thing2",
		})
		So(result.Associations[1].DataBag, ShouldResemble, map[string]interface{}{})
	})
}

func TestNewListAssocResponseWithMeta(t *testing.T) {
	Convey("When creating a new ListAssocResponseWithMeta", t, func() {
		now := time.Now()
		bytes, _ := json.Marshal(map[string]interface{}{})
		resp := &rpc.GetAllResponseV2{
			From: &rpc.Entity{Name: "123", Kind: "user"},
			To: []*rpc.AssocResponseWithMeta{
				&rpc.AssocResponseWithMeta{
					Kind: "relation_a",
					Target: &rpc.AssocResponse{
						Target: &rpc.Entity{Name: "456", Kind: "user"},
						Data:   bytes,
						Time:   &rpc.Timestamp{Seconds: now.Unix(), Nanos: int64(now.Nanosecond())},
					},
				},
				&rpc.AssocResponseWithMeta{
					Kind: "relation_b",
					Target: &rpc.AssocResponse{
						Target: &rpc.Entity{Name: "789", Kind: "user"},
						Time:   &rpc.Timestamp{Seconds: now.Unix(), Nanos: int64(now.Nanosecond())},
					},
				},
			},
		}

		result := newListAssocResponseWithMeta(resp)

		So(result.From, ShouldResemble, Entity{ID: "123", Kind: "user"})
		So(len(result.Associations), ShouldEqual, len(resp.To))
		assocA := result.Associations[0]
		So(assocA.Kind, ShouldEqual, "relation_a")
		So(assocA.Assoc.Entity.ID, ShouldEqual, "456")

		assocB := result.Associations[1]
		So(assocB.Kind, ShouldEqual, "relation_b")
		So(assocB.Assoc.Entity.ID, ShouldEqual, "789")
	})
}
