package partnerships

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"golang.org/x/net/context"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/partnerships/models"
)

func TestGetPropertiesByUserID(t *testing.T) {
	Convey("when calling GetPropertiesByUserID", t, func() {
		ctx := context.Background()
		userID := 45573555

		Convey("when the backend is available", func() {
			partnerProgram := true
			properties := &models.PartnerProperties{
				UserID:         &userID,
				PartnerProgram: &partnerProgram,
			}

			ts := initTestServer(userID, properties)
			defer ts.Close()

			c, err := NewClient(twitchhttp.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetPropertiesByUserID(ctx, userID, nil)
			So(err, ShouldBeNil)
			So(returnedProperties, ShouldResemble, properties)
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchhttp.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetPropertiesByUserID(ctx, userID, nil)
			So(err, ShouldNotBeNil)
			So(returnedProperties, ShouldBeNil)
		})
	})
}

func TestCheckPartneredByUserID(t *testing.T) {
	Convey("when calling CheckPartneredByUserID", t, func() {
		ctx := context.Background()
		userID := 45573555

		Convey("when the backend is available", func() {
			partnerProgram := true
			properties := &models.PartnerProperties{
				UserID:         &userID,
				PartnerProgram: &partnerProgram,
			}

			ts := initTestServer(userID, properties)
			defer ts.Close()

			c, err := NewClient(twitchhttp.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			Convey("with a partnered user", func() {
				partnered, err := c.CheckPartneredByUserID(ctx, userID, nil)
				So(err, ShouldBeNil)
				So(partnered, ShouldEqual, true)
			})

			Convey("with an unpartnered user", func() {
				partnerProgram = false
				partnered, err := c.CheckPartneredByUserID(ctx, userID, nil)
				So(err, ShouldBeNil)
				So(partnered, ShouldEqual, false)
			})

			Convey("with nil properties", func() {
				properties.PartnerProgram = nil
				partnered, err := c.CheckPartneredByUserID(ctx, userID, nil)
				So(err, ShouldBeNil)
				So(partnered, ShouldEqual, false)
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchhttp.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			partnered, err := c.CheckPartneredByUserID(ctx, userID, nil)
			So(err, ShouldNotBeNil)
			So(partnered, ShouldEqual, false)
		})
	})
}

func initBrokenTestServer() *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))
}

func initTestServer(userID int, properties *models.PartnerProperties) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != fmt.Sprintf("/partners/%d/properties", userID) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		err := json.NewEncoder(w).Encode(properties)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}))
}
