package main

import (
	"fmt"
	"log"
	"strconv"
	"time"

	_ "github.com/lib/pq"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/identity/rediser"
	"code.justin.tv/web/partnerships/api"
	"code.justin.tv/web/partnerships/backend"
	"code.justin.tv/web/partnerships/cache"
	"code.justin.tv/web/partnerships/database"
)

const rediserMetricsSampleRate = 0.1

func main() {
	config.Register(map[string]string{
		// shared config for master and slave
		"db-user":     "",
		"db-password": "",
		"db-name":     "justintv_dev",

		// slave config
		"db-host":                    "master-sitedb.staging.sfo01.justin.tv",
		"db-port":                    "6543",
		"db-max-open-conns":          "100",
		"db-max-idle-conns":          "50",
		"db-max-queue-size":          "10",
		"db-conn-acquire-timeout-ms": "1000",
		"db-request-timeout-ms":      "5000",
		"db-max-conn-age-ms":         "60000",

		// master config
		"db-master-host":                    "master-sitedb.staging.sfo01.justin.tv",
		"db-master-port":                    "6543",
		"db-master-max-open-conns":          "100",
		"db-master-max-idle-conns":          "50",
		"db-master-max-queue-size":          "10",
		"db-master-conn-acquire-timeout-ms": "1000",
		"db-master-request-timeout-ms":      "5000",
		"db-master-max-conn-age-ms":         "60000",

		// redis cache
		"redis-host":             "staging-partnerships.qg4ybk.clustercfg.usw2.cache.amazonaws.com:6379",
		"redis-pool-size":        "100",
		"redis-timeout-dial-ms":  "1000",
		"redis-timeout-read-ms":  "200",
		"redis-timeout-write-ms": "200",
	})

	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	slaveDB, err := configureSlaveDB()
	if err != nil {
		log.Fatal(err)
	}

	masterDB, err := configureMasterDB()
	if err != nil {
		log.Fatal(err)
	}

	slaveDBQuerier, err := configureDbQuerier(slaveDB)
	if err != nil {
		log.Fatal(err)
	}

	masterDBQuerier, err := configureDbQuerier(masterDB)
	if err != nil {
		log.Fatal(err)
	}

	cacher, err := configureCacher()
	if err != nil {
		log.Fatal(err)
	}

	partnerPropertiesRepository, err := configurePartnerPropertiesRepository(slaveDBQuerier, masterDBQuerier, cacher)
	if err != nil {
		log.Fatal(err)
	}

	server, err := api.NewServer(partnerPropertiesRepository)
	if err != nil {
		log.Fatal(err)
	}

	log.Fatal(twitchhttp.ListenAndServe(server))
}

func configureSlaveDB() (db.DB, error) {
	dbConn, err := db.Open(
		db.DriverName("postgres"),
		db.Host(config.Resolve("db-host")),
		db.Port(configResolveInt("db-port")),
		db.User(config.Resolve("db-user")),
		db.Password(config.Resolve("db-password")),
		db.DBName(config.Resolve("db-name")),
		db.MaxOpenConns(configResolveInt("db-max-open-conns")),
		db.MaxIdleConns(configResolveInt("db-max-idle-conns")),
		db.MaxQueueSize(configResolveInt("db-max-queue-size")),
		db.ConnAcquireTimeout(configResolveDuration("db-conn-acquire-timeout-ms")),
		db.RequestTimeout(configResolveDuration("db-request-timeout-ms")),
		db.MaxConnAge(configResolveDuration("db-max-conn-age-ms")),
	)
	if err != nil {
		return nil, err
	}

	err = configureDBStats(dbConn, "db")
	if err != nil {
		return nil, err
	}

	return dbConn, nil
}

func configureMasterDB() (db.DB, error) {
	dbConn, err := db.Open(
		db.DriverName("postgres"),
		db.Host(config.Resolve("db-master-host")),
		db.Port(configResolveInt("db-master-port")),
		db.User(config.Resolve("db-user")),
		db.Password(config.Resolve("db-password")),
		db.DBName(config.Resolve("db-name")),
		db.MaxOpenConns(configResolveInt("db-master-max-open-conns")),
		db.MaxIdleConns(configResolveInt("db-master-max-idle-conns")),
		db.MaxQueueSize(configResolveInt("db-master-max-queue-size")),
		db.ConnAcquireTimeout(configResolveDuration("db-master-conn-acquire-timeout-ms")),
		db.RequestTimeout(configResolveDuration("db-master-request-timeout-ms")),
		db.MaxConnAge(configResolveDuration("db-master-max-conn-age-ms")),
	)
	if err != nil {
		return nil, err
	}

	err = configureDBStats(dbConn, "db-master")
	if err != nil {
		return nil, err
	}

	return dbConn, nil
}

func configureDBStats(dbConn db.DB, namespace string) error {
	logger, err := database.NewLogger(config.Statsd(), fmt.Sprintf("%s.", namespace))
	if err != nil {
		return err
	}

	dbConn.SetCallbacks(logger.LogDBStat, logger.LogRunStat)

	go func() {
		ticker := time.Tick(10 * time.Second)
		for {
			select {
			case <-ticker:
				logger.LogDBState(dbConn.Info())
			}
		}
	}()

	return nil
}

func configResolveInt(field string) int {
	i, err := strconv.Atoi(config.Resolve(field))
	if err != nil {
		log.Fatal(fmt.Sprintf("config parameter '%s' is not valid: ", field), err)
	}
	return i
}

func configResolveDuration(field string) time.Duration {
	i := configResolveInt(field)
	return time.Duration(i) * time.Millisecond
}

func configureDbQuerier(db db.DB) (database.Querier, error) {
	return database.NewQuerier(db)
}

func configurePartnerPropertiesRepository(slaveDB database.Querier, masterDB database.Querier, cacher cache.Cacher) (backend.PartnerPropertiesRepository, error) {
	return backend.NewPartnerPropertiesRepository(slaveDB, masterDB, cacher)
}

func configureCacher() (cache.Cacher, error) {
	opts := &rediser.Options{
		Addrs:           []string{config.Resolve("redis-host")},
		KeyPrefix:       "partnerships",
		StatPrefix:      "partnerships_redis",
		StatSampleRate:  rediserMetricsSampleRate,
		MonitorInterval: 5 * time.Second,
		PoolSize:        configResolveInt("redis-pool-size"),
		DialTimeout:     time.Duration(configResolveInt("redis-timeout-dial-ms")) * time.Millisecond,
		ReadTimeout:     time.Duration(configResolveInt("redis-timeout-read-ms")) * time.Millisecond,
		WriteTimeout:    time.Duration(configResolveInt("redis-timeout-write-ms")) * time.Millisecond,
	}

	client, err := rediser.NewCache(opts, config.Statsd())
	if err != nil {
		return nil, err
	}

	return client, nil
}
