package twitter

import (
	"encoding/json"
	"fmt"
	"net/url"
	"strings"
	"sync"
	"unicode"
	"unicode/utf8"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/identity/connections/client"

	"github.com/kylemcc/twitter-text-go/extract"
	"github.com/mrjones/oauth"
	"golang.org/x/net/context"
	"golang.org/x/text/unicode/norm"
)

const (
	TWITTER_CHARACTER_LENGTH = 140

	TWITTER_CONFIG_URL        = "https://api.twitter.com/1.1/help/configuration.json"
	TWITTER_UPDATE_STATUS_URL = "https://api.twitter.com/1.1/statuses/update.json"

	// These are defined, but should likely not be used, we use the connections
	// service for managing these connections
	TWITTER_OAUTH_REQUEST_TOKEN_URL   = "https://api.twitter.com/oauth/request_token"
	TWITTER_OAUTH_AUTHORIZE_TOKEN_URL = "https://api.twitter.com/oauth/authorize"
	TWITTER_OAUTH_ACCESS_TOKEN_URL    = "https://api.twitter.com/oauth/access_token"

	// BASE URL FOR A TWEET fmt.Sprintf(BASE, screen_name, tweet_id)
	TWITTER_TWEET_URL_BASE = "https://twitter.com/%s/status/%d"
)

var (
	ErrFailedToLoadConfig = errors.New("Failed to load config from twitter, possibly rate limited")
)

type Client struct {
	ConnectionsClient connections.Client
	twitterConfig     *TwitterConfig
	configInit        sync.Mutex
	consumer          *oauth.Consumer
}

type TwitterConfig struct {
	Loaded                  bool `json:"-"`
	CharactersForMedia      int  `json:"characters_reserved_per_media"`
	CharactersForShortHTTP  int  `json:"short_url_length"`
	CharactersForShortHTTPS int  `json:"short_url_length_https"`
}

type twitterUser struct {
	ID         int    `json:"id"`
	ScreenName string `json:"screen_name"`
}

type twitterTweet struct {
	ID   int         `json:"id"`
	User twitterUser `json:"user"`
}

type twitterErrorResponse struct {
	Errors []TwitterError `json:"errors"`
}

type TwitterError struct {
	Message string `json:"message"`
	Code    int    `json:"code"`
}

func (e TwitterError) Error() string {
	return fmt.Sprintf("Code: %d, Message: %s", e.Code, e.Message)
}

func NewClient(consumerKey, consumerSecret string, connectionsClient connections.Client) *Client {
	twitterConsumer := oauth.NewConsumer(
		consumerKey,
		consumerSecret,
		oauth.ServiceProvider{
			RequestTokenUrl:   TWITTER_OAUTH_REQUEST_TOKEN_URL,
			AuthorizeTokenUrl: TWITTER_OAUTH_AUTHORIZE_TOKEN_URL,
			AccessTokenUrl:    TWITTER_OAUTH_ACCESS_TOKEN_URL,
		},
	)

	return &Client{
		ConnectionsClient: connectionsClient,
		twitterConfig:     &TwitterConfig{Loaded: false},
		consumer:          twitterConsumer,
	}
}

// PostLinkForTwitchUserID
// This guarantees that our twitter config is fresh, loads the connections for
// the user and then attempts to truncate and post a tweet with the included
// link appended.
// Returns a link to the created tweet and/or errors encountered while trying
// to create the post.
func (c *Client) PostLinkForTwitchUserID(ctx context.Context, status string, url string, userID string) (string, error) {
	twitterUser, err := c.ConnectionsClient.GetTwitterUser(ctx, userID, nil)
	if err != nil {
		return "", err
	}
	token := &oauth.AccessToken{
		Token:  twitterUser.AccessToken,
		Secret: twitterUser.AccessSecret,
	}

	if !c.twitterConfig.Loaded {
		c.configInit.Lock()
		err := c.loadTwitterConfig(ctx, token)
		c.configInit.Unlock()
		if err != nil {
			return "", ErrFailedToLoadConfig
		}
	}

	maxLength := TWITTER_CHARACTER_LENGTH - (c.twitterConfig.CharactersForShortHTTPS + 1)
	truncatedMessage := c.truncateMessage(status, maxLength, "…")

	message := strings.Join([]string{truncatedMessage, url}, " ")
	return c.postStatus(ctx, message, token)
}

func (c *Client) loadTwitterConfig(ctx context.Context, token *oauth.AccessToken) error {
	client, err := c.consumer.MakeHttpClient(token)
	if err != nil {
		return err
	}

	response, err := client.Get(TWITTER_CONFIG_URL)
	if err != nil {
		return err
	}

	dec := json.NewDecoder(response.Body)
	err = dec.Decode(c.twitterConfig)
	if err != nil {
		return err
	}

	c.twitterConfig.Loaded = true
	return nil
}

func (c *Client) postStatus(ctx context.Context, status string, token *oauth.AccessToken) (string, error) {
	client, err := c.consumer.MakeHttpClient(token)
	if err != nil {
		return "", err
	}

	response, err := client.PostForm(TWITTER_UPDATE_STATUS_URL, url.Values{"status": []string{status}})
	if err != nil {
		return "", err
	}

	dec := json.NewDecoder(response.Body)
	if response.StatusCode != 200 {
		twitterError := &twitterErrorResponse{}
		err = dec.Decode(twitterError)
		if err != nil || len(twitterError.Errors) == 0 {
			return "", fmt.Errorf("Unexpected status code returned from API %d", response.StatusCode)
		}
		return "", errors.Wrap(twitterError.Errors[0], fmt.Sprintf("unexpected http code returned from API %d", response.StatusCode))
	}
	tweet := &twitterTweet{}
	err = dec.Decode(tweet)
	if err != nil {
		return "", err
	}

	return fmt.Sprintf(TWITTER_TWEET_URL_BASE, tweet.User.ScreenName, tweet.ID), nil
}

// Custom truncation that does not break words. This will instead fit as many
// whitespace delimited tokens as it can before truncating and adding an ellipsis
func (c *Client) truncateMessage(status string, maxLength int, continuationString string) string {

	tokens := c.tokenize(status)
	truncatedTokens := []string{}
	truncated := false

	maxLengthWithContinuation := maxLength - c.getStringLength(continuationString)
	totalTokenLength := 0

	for _, token := range tokens {
		totalTokenLength += c.getStringLength(token)

		if totalTokenLength <= maxLengthWithContinuation {
			truncatedTokens = append(truncatedTokens, token)
		}

		if totalTokenLength > maxLength {
			truncated = true
			break
		}
	}

	if !truncated {
		return status
	}

	truncatedString := strings.Join(truncatedTokens, "")

	// Trim trailing whitespace because appending ellipsis to whitespace looks weird.
	truncatedString = strings.TrimRightFunc(truncatedString, unicode.IsSpace)
	return truncatedString + continuationString
}

// tokenize splits a string on whitespace into tokens.
// tokenize includes whitespace as tokens to facilitate preserving newlines and multiple spaces when truncating the
// status.
func (c *Client) tokenize(status string) []string {
	tokens := make([]string, 0, 0)

	token := make([]rune, 0, 0)
	prevIsSpace := false

	for _, r := range status {
		if unicode.IsSpace(r) || prevIsSpace {
			if len(token) > 0 {
				tokens = append(tokens, string(token))
				token = make([]rune, 0, 0)
			}
		}

		token = append(token, r)
		prevIsSpace = unicode.IsSpace(r)
	}

	if len(token) > 0 {
		tokens = append(tokens, string(token))
	}

	return tokens
}

// getTokenLength takes a token (a non-whitespace blob, or a whitespace char), and returns the number of characters
// that Twitter would find in the token.
func (c *Client) getStringLength(token string) int {

	normalizedToken := norm.NFC.String(token)
	length := utf8.RuneCountInString(normalizedToken)

	lowerCaseToken := strings.ToLower(normalizedToken)
	urls := extract.ExtractUrls(lowerCaseToken)
	for _, url := range urls {
		length -= url.Range.Length()
		if strings.HasPrefix(url.Text, "https://") {
			length += c.twitterConfig.CharactersForShortHTTPS
		} else {
			length += c.twitterConfig.CharactersForShortHTTP
		}
	}

	return length
}
