package twitter

import (
	"testing"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/web/twitter/client/mocks"
)

const (
	SHORT_HTTP_LINK_LENGTH  = 23
	SHORT_HTTPS_LINK_LENGTH = 23
)

func TestSpec(t *testing.T) {
	Convey("Given a client", t, func() {
		mockConnectionsClient := &mocks.Client{}
		client := NewClient("", "", mockConnectionsClient)
		client.twitterConfig = &TwitterConfig{
			Loaded:                  true,
			CharactersForMedia:      0,
			CharactersForShortHTTP:  SHORT_HTTP_LINK_LENGTH,
			CharactersForShortHTTPS: SHORT_HTTPS_LINK_LENGTH,
		}

		So(client, ShouldNotBeNil)

		Convey("TruncateMessage", func() {

			testCases := []truncateMessageTestCase{
				{
					description:    "When length of the status is less than the max length",
					status:         "a b c",
					maxLength:      6,
					continuation:   "…",
					expectedResult: "a b c",
				}, {
					description:    "When length of the status is equal to the max length",
					status:         "a b c",
					maxLength:      5,
					continuation:   "…",
					expectedResult: "a b c",
				}, {
					description:    "When length of the status is greater than the max length, and a non-whitespace token starts at max length",
					status:         "aa bb",
					maxLength:      3,
					continuation:   "…",
					expectedResult: "aa…",
				}, {
					description:    "When length of the status is greater than the max length, and a non-whitespace token overlaps the max length",
					status:         "a bb",
					maxLength:      3,
					continuation:   "…",
					expectedResult: "a…",
				}, {
					description:    "When there is a long continuation string",
					status:         "aa bbb cc dd",
					maxLength:      10,
					continuation:   "... zz",
					expectedResult: "aa... zz",
				}, {
					description:    "When status contains newlines",
					status:         "a b\nc",
					maxLength:      4,
					continuation:   "…",
					expectedResult: "a b…",
				},
			}

			for _, testCase := range testCases {
				testTruncateMessage(client, testCase)
			}
		})

		Convey("getTokenLength", func() {

			Convey("When token is a word", func() {
				token := "apples"

				Convey("returns the length of a shortened HTTP link", func() {
					l := client.getStringLength(token)
					So(l, ShouldEqual, 6)
				})
			})

			Convey("When token is an HTTP URL", func() {
				token := "http://www.google.com"

				Convey("returns the length of a shortened HTTP link", func() {
					l := client.getStringLength(token)
					So(l, ShouldEqual, SHORT_HTTP_LINK_LENGTH)
				})
			})

			Convey("When token is an HTTPS URL", func() {
				token := "https://www.google.com"

				Convey("returns the length of a shortened HTTPS link", func() {
					l := client.getStringLength(token)
					So(l, ShouldEqual, SHORT_HTTPS_LINK_LENGTH)
				})
			})

			Convey("When token is an incomplete URL", func() {
				token := "google.com"

				Convey("returns the length of a shortened HTTP link", func() {
					l := client.getStringLength(token)
					So(l, ShouldEqual, SHORT_HTTP_LINK_LENGTH)
				})
			})

			Convey("When token is a URL with upper case letters in the suffix", func() {
				token := "google.Com"

				Convey("returns the length of a shortened HTTP link", func() {
					l := client.getStringLength(token)
					So(l, ShouldEqual, SHORT_HTTP_LINK_LENGTH)
				})
			})

			Convey("When token is an email address that looks like it contains a URL", func() {
				token := "a.com" + ".a@gmail.com"

				Convey("returns the length of the URL + the length of the remainder of the address", func() {
					l := client.getStringLength(token)
					So(l, ShouldEqual, SHORT_HTTP_LINK_LENGTH+12)
				})
			})
		})

		Convey("tokenize", func() {

			testCases := []tokenizeTestCase{
				{
					description:    "when string is empty",
					status:         "",
					expectedTokens: []string{},
				}, {
					description:    "when string has a single non-whitespace token",
					status:         "a",
					expectedTokens: []string{"a"},
				}, {
					description:    "when string has a non-whitespace token followed by whitespace",
					status:         "a ",
					expectedTokens: []string{"a", " "},
				}, {
					description:    "when string consists of multiple whitespace tokens",
					status:         "  ",
					expectedTokens: []string{" ", " "},
				}, {
					description:    "when string consists of a multi-character token",
					status:         "ab",
					expectedTokens: []string{"ab"},
				}, {
					description:    "when string consists of non-whitespace tokens separated by spaces",
					status:         "ab cc",
					expectedTokens: []string{"ab", " ", "cc"},
				}, {
					description:    "when string consists of non-whitespace tokens separated by newlines",
					status:         "ab\ncc",
					expectedTokens: []string{"ab", "\n", "cc"},
				},
			}

			for _, testCase := range testCases {
				testTokenize(client, testCase)
			}

		})
	})
}

type truncateMessageTestCase struct {
	description    string
	status         string
	maxLength      int
	continuation   string
	expectedResult string
}

func testTruncateMessage(client *Client, testCase truncateMessageTestCase) {
	Convey(testCase.description, func() {
		truncatedMessage := client.truncateMessage(testCase.status, testCase.maxLength, testCase.continuation)
		So(truncatedMessage, ShouldEqual, testCase.expectedResult)
	})
}

type tokenizeTestCase struct {
	description    string
	status         string
	expectedTokens []string
}

func testTokenize(client *Client, testCase tokenizeTestCase) {
	Convey(testCase.description, func() {
		tokens := client.tokenize(testCase.status)
		So(tokens, ShouldResemble, testCase.expectedTokens)
	})
}
