package models

import (
	"fmt"
	"time"

	"code.justin.tv/common/yimg"
)

type ChannelProperties struct {
	ID                  uint64     `json:"id"`
	Name                string     `json:"name"`
	DirectoryHidden     bool       `json:"directory_hidden"`
	Broadcaster         *string    `json:"broadcaster"`
	BroadcasterLanguage *string    `json:"broadcaster_language"`
	BroadcasterSoftware *string    `json:"broadcaster_software"`
	CreatedOn           *time.Time `json:"created_on"`
	Game                *string    `json:"game"`
	GameID              *uint64    `json:"game_id"`
	Mature              *bool      `json:"mature"`
	Status              *string    `json:"status"`
	Title               *string    `json:"title"`
	UpdatedOn           *time.Time `json:"updated_on"`
	ViewsCount          *uint64    `json:"views_count"`
	Delay               *uint64    `json:"delay"`

	BlockNonPublicAds *bool `json:"block_non_public_ads"`
	PrerollsDisabled  *bool `json:"prerolls_disabled"`
	PostrollsDisabled *bool `json:"postrolls_disabled"`
	FightAdBlock      *bool `json:"fight_ad_block"`

	CanRebroadcast           *bool      `json:"can_rebroadcast"`
	DelayControlsEnabled     *bool      `json:"delay_controls_enabled"`
	LastBroadcastTime        *time.Time `json:"last_broadcast_time"`
	LastBroadcastID          *string    `json:"last_broadcast_id"`
	LastLiveNotificationSent *time.Time `json:"last_live_notification_sent"`

	CanCreateOfflinePlaylist *bool   `json:"can_create_offline_playlist"`
	About                    *string `json:"about"`
	RedirectChannel          *string `json:"redirect_channel"`
	PrimaryTeamID            *uint64 `json:"primary_team_id"`
	DisableChat              *bool   `json:"disable_chat"`

	ChannelOfflineImage          yimg.Images `json:"channel_offline_image"`
	ProfileBanner                yimg.Images `json:"profile_banner"`
	ProfileBannerBackgroundColor *string     `json:"profile_banner_background_color"`
}

type UpdateChannelProperties struct {
	ID                  uint64     `json:"id" validate:"nonzero"`
	DirectoryHidden     *bool      `json:"directory_hidden"`
	Broadcaster         *string    `json:"broadcaster"`
	BroadcasterLanguage *string    `json:"broadcaster_language"`
	BroadcasterSoftware *string    `json:"broadcaster_software"`
	Game                *string    `json:"game"`
	GameID              *uint64    `json:"game_id"`
	Mature              *bool      `json:"mature"`
	Status              *string    `json:"status"`
	Title               *string    `json:"title"`
	UpdatedOn           *time.Time `json:"updated_on"`
	ViewsCount          *uint64    `json:"views_count"`
	Delay               *uint64    `json:"delay"`

	BlockNonPublicAds *bool `json:"block_non_public_ads"`
	PrerollsDisabled  *bool `json:"prerolls_disabled"`
	PostrollsDisabled *bool `json:"postrolls_disabled"`
	FightAdBlock      *bool `json:"fight_ad_block"`

	CanRebroadcast           *bool      `json:"can_rebroadcast"`
	DelayControlsEnabled     *bool      `json:"delay_controls_enabled"`
	LastBroadcastTime        *time.Time `json:"last_broadcast_time"`
	LastBroadcastID          *string    `json:"last_broadcast_id"`
	LastLiveNotificationSent *time.Time `json:"last_live_notification_sent"`

	CanCreateOfflinePlaylist *bool   `json:"can_create_offline_playlist"`
	About                    *string `json:"about"`
	RedirectChannel          *string `json:"redirect_channel"`
	PrimaryTeamID            *uint64 `json:"primary_team_id"`
	DisableChat              *bool   `json:"disable_chat"`
}

func (up *UpdateChannelProperties) FillFromProperties(p *ChannelProperties) {
	if up.DirectoryHidden == nil {
		up.DirectoryHidden = &p.DirectoryHidden
	}

	if up.Broadcaster == nil {
		up.Broadcaster = p.Broadcaster
	}

	if up.BroadcasterLanguage == nil {
		up.BroadcasterLanguage = p.BroadcasterLanguage
	}

	if up.BroadcasterSoftware == nil {
		up.BroadcasterSoftware = p.BroadcasterSoftware
	}

	if up.Game == nil {
		up.Game = p.Game
	}

	if up.GameID == nil {
		up.GameID = p.GameID
	}

	if up.Mature == nil {
		up.Mature = p.Mature
	}

	if up.Status == nil {
		up.Status = p.Status
	}

	if up.Title == nil {
		up.Title = p.Title
	}

	if up.ViewsCount == nil {
		up.ViewsCount = p.ViewsCount
	}

	if up.Delay == nil {
		up.Delay = p.Delay
	}

	if up.BlockNonPublicAds == nil {
		up.BlockNonPublicAds = p.BlockNonPublicAds
	}

	if up.PrerollsDisabled == nil {
		up.PrerollsDisabled = p.PrerollsDisabled
	}

	if up.PostrollsDisabled == nil {
		up.PostrollsDisabled = p.PostrollsDisabled
	}

	if up.FightAdBlock == nil {
		up.FightAdBlock = p.FightAdBlock
	}

	if up.CanRebroadcast == nil {
		up.CanRebroadcast = p.CanRebroadcast
	}

	if up.DelayControlsEnabled == nil {
		up.DelayControlsEnabled = p.DelayControlsEnabled
	}

	if up.LastBroadcastTime == nil {
		up.LastBroadcastTime = p.LastBroadcastTime
	}

	if up.LastBroadcastID == nil {
		up.LastBroadcastID = p.LastBroadcastID
	}

	if up.LastLiveNotificationSent == nil {
		up.LastLiveNotificationSent = p.LastLiveNotificationSent
	}

	if up.CanCreateOfflinePlaylist == nil {
		up.CanCreateOfflinePlaylist = p.CanCreateOfflinePlaylist
	}

	if up.About == nil {
		up.About = p.About
	}

	if up.RedirectChannel == nil {
		up.RedirectChannel = p.RedirectChannel
	}

	if up.PrimaryTeamID == nil {
		up.PrimaryTeamID = p.PrimaryTeamID
	}

	if up.DisableChat == nil {
		up.DisableChat = p.DisableChat
	}
}

const (
	ChannelsIDCacheKey    = "u.id"
	ChannelsLoginCacheKey = "login"
)

func ChannelIDToString(id uint64) string {
	return fmt.Sprintf("%d", id)
}

func (p ChannelProperties) CachePairs() []CachePair {
	var pairs []CachePair

	if p.ID != 0 {
		pairs = append(pairs, CachePair{
			Key:   ChannelsIDCacheKey,
			Value: ChannelIDToString(p.ID),
		})
	}

	if p.Name != "" {
		pairs = append(pairs, CachePair{
			Key:   ChannelsLoginCacheKey,
			Value: p.Name,
		})
	}

	return pairs
}

type ChannelPropertiesIterator []ChannelProperties

func (i ChannelPropertiesIterator) Each(f func(Cacheable) error) error {
	for _, prop := range i {
		if err := f(prop); err != nil {
			return err
		}
	}

	return nil
}
