// Package extract provides a set of routines for extracting various Twitter
// "entities" from text
//
// This package supports extraction of Twitter usernames, replies, lists,
// hashtags, cashtags, and URLs. The implementation and API are based the
// set of similarly named twitter-text-* libraries published by Twitter. This
// library is tested using the standard Conformance test suite maintained by
// Twitter (https://github.com/twitter/twitter-text-conformance).
package extract

import (
	"fmt"
	"sort"
	"strings"
	"unicode/utf8"
)

// Used to specify offsets in a string where entities
// are located
type Range struct {
	Start int
	Stop  int
}

// Returns the length of the range
func (r *Range) Length() int {
	return r.Stop - r.Start
}

// Implement the Stringer interface
func (r Range) String() string {
	return fmt.Sprintf("(%d, %d)", r.Start, r.Stop)
}

// Specifies the type of a given entity
type EntityType int

const (
	MENTION EntityType = iota
	HASH_TAG
	CASH_TAG
	URL
)

// Implement the Stringer interface
func (t EntityType) String() string {
	switch t {
	case MENTION:
		return "MENTION"
	case HASH_TAG:
		return "HASH_TAG"
	case CASH_TAG:
		return "CASH_TAG"
	case URL:
		return "URL"
	}
	return "Unknown"
}

// Structure representing an "entity" within a
// string.
type TwitterEntity struct {
	Text      string // The value of the entity
	Range     Range  // Represents the location of the entity in character/rune offsets
	ByteRange Range  // Represents the location of the entity in byte offsets
	Type      EntityType

	screenName string // Contains the value of username without the leading '@' when Type=MENTION
	listSlug   string // Contains the value of the list name when Type=MENTION
	hashtag    string // Contains the value of the hashtag without the leading # when Type=HASH_TAG
	cashtag    string // Contains the value of the cashtag without the leading $ when Type=CASH_TAG

	screenNameIsSet bool
	listSlugIsSet   bool
	hashtagIsSet    bool
	cashtagIsSet    bool
}

type entitiesT []*TwitterEntity

// The indices generated by the various extract functions are
// byte offsets. This function calculates charecter/rune offsets
// based on those offsets
func (entities entitiesT) fixIndices(text string) {
	for _, e := range entities {
		start := utf8.RuneCountInString(text[:e.ByteRange.Start])
		e.Range.Start = start
		stop := utf8.RuneCountInString(e.Text)
		e.Range.Stop = start + stop
	}
}

func (e *entitiesT) removeOverlappingEntities() {
	if e.Len() < 2 {
		return
	}

	var (
		result   []*TwitterEntity
		prev     *TwitterEntity
		cur      *TwitterEntity
		entities = *e
	)
	prev = entities[0]
	result = append(result, prev)
	for _, cur = range entities[1:] {
		if !(prev.Range.Stop > cur.Range.Start) {
			result = append(result, cur)
		}
		prev = cur
	}
	*e = result
}

func (e entitiesT) Len() int {
	return len(e)
}

func (e entitiesT) Less(i, j int) bool {
	return e[i].Range.Start < e[j].Range.Start
}

func (e entitiesT) Swap(i, j int) {
	e[i], e[j] = e[j], e[i]
}

// Implement the Stringer interface
func (t *TwitterEntity) String() string {
	return fmt.Sprintf("TwitterEntity{Text: [%s] Range: %+v Type: %v", t.Text, t.Range, t.Type)
}

// Returns the value of the extracted screen name (when Type=MENTION) and
// a boolean indicating whether the value is set. The return value will be
// ("", false) when Type != MENTION
func (t *TwitterEntity) ScreenName() (string, bool) {
	return t.screenName, t.screenNameIsSet
}

// Returns the value of the extracted list name (when Type=MENTION) and
// a boolean indicating whether the value is set. The return value will be
// ("", false) when Type != MENTION OR when the extracted entity is simply
// a mention and not a list name
func (t *TwitterEntity) ListSlug() (string, bool) {
	return t.listSlug, t.listSlugIsSet
}

// Returns the value of the extracted hashtag (when Type=HASH_TAG) and
// a boolean indicating whether the value is set. The return value will be
// ("", false) when Type != HASH_TAG
func (t *TwitterEntity) Hashtag() (string, bool) {
	return t.hashtag, t.hashtagIsSet
}

// Returns the value of the extracted cashtag (when Type=CASH_TAG) and
// a boolean indicating whether the value is set. The return value will be
// ("", false) when Type != CASH_TAG
func (t *TwitterEntity) Cashtag() (string, bool) {
	return t.cashtag, t.cashtagIsSet
}

// Extract all usernames, lists, hashtags, and URLs from the
// given text - returned in the order they appear within the
// input string
func ExtractEntities(text string) []*TwitterEntity {
	var result entitiesT
	result = ExtractUrls(text)
	result = append(result, ExtractHashtags(text)...)
	result = append(result, ExtractMentionsOrLists(text)...)
	result = append(result, ExtractCashtags(text)...)

	sort.Sort(result)
	result.removeOverlappingEntities()
	return result
}

// Extract urls from the given text. Returns a slice of
// TwitterEntity struct pointers.
func ExtractUrls(text string) []*TwitterEntity {
	// This giant pile of barf is copied from the various
	// twitter-text implementations. There must be a better
	// way!
	var result entitiesT
	var (
		matchStart     int
		matchEnd       int
		precedingStart int
		precedingEnd   int
		domainStart    int
		domainEnd      int
		pathStart      int
		pathEnd        int
	)

	// Start at the beginning of the input string,
	// walking forward one match at a time. We have
	// to walk the string because the regexp package
	// lacks support for lookahead assertions
	offset := 0
	nextOffset := 0
	for {
		offset = nextOffset
		substr := text[offset:]
		match := validUrl.FindStringSubmatchIndex(substr)

		// If no matches are found in this portion of the string,
		// we're done
		if match == nil {
			break
		}

		// Next time around, start at the end of the current match,
		// minus 1 because indices are not inclusive
		nextOffset = match[1] + offset - 1

		matchStart = match[validUrlGroupUrl*2]
		matchEnd = match[validUrlGroupUrl*2+1]

		// If protocol is missing, only extract ascii domains
		if match[validUrlGroupProtocol*2] < 0 {
			var lastEntity *TwitterEntity
			lastInvalid := false
			precedingStart = match[validUrlGroupBefore*2]
			precedingEnd = match[validUrlGroupBefore*2+1]
			domainStart = match[validUrlGroupDomain*2]
			domainEnd = match[validUrlGroupDomain*2+1]
			pathStart = match[validUrlGroupPath*2]
			pathEnd = match[validUrlGroupPath*2+1]

			// check for invalid preceding character
			if invalidUrlWithoutProtocolMatchBegin.MatchString(substr[precedingStart:precedingEnd]) {
				continue
			}

			// Make sure the protocol-less domain is ascii only
			// e.g., in the case of "한국twitter.com", only extract twitter.com
			if m := validAsciiDomain.FindStringSubmatchIndex(substr[domainStart:domainEnd]); m != nil {
				lastEntity = &TwitterEntity{
					Text: substr[matchStart+m[0] : matchStart+m[1]],
					ByteRange: Range{
						Start: matchStart + offset + m[0],
						Stop:  matchStart + offset + m[1]},
					Type: URL}

				// Set the next offset to the end of this match
				nextOffset = matchStart + m[1] + offset - 1

				// If the url has a Generic TLD (not CC TLD), it's valid
				if lastInvalid = invalidShortDomain.MatchString(lastEntity.Text); !lastInvalid {
					result = append(result, lastEntity)
				}
			}

			if lastEntity == nil {
				continue
			}

			// If the match contains a path immediately following the domain,
			// append it to the match
			if pathStart > 0 && pathStart == lastEntity.ByteRange.Stop-offset {
				// If the last result was invalid b/c it did not contain a GTLD,
				// append it
				if lastInvalid {
					result = append(result, lastEntity)
				}

				// Update the text and offsets
				lastEntity.Text += substr[pathStart:pathEnd]
				lastEntity.ByteRange.Stop = pathEnd + offset
				nextOffset = lastEntity.ByteRange.Stop - 1
			} else if validSpecialShortDomain.MatchString(lastEntity.Text) {
				result = append(result, lastEntity)
			}
		} else {
			// Else, the url contains a protocol
			url := substr[matchStart:matchEnd]
			// If it's a t.co url, restrict to certain path characters
			if tcoLoc := validTcoUrl.FindStringIndex(url); tcoLoc != nil {
				url = url[tcoLoc[0]:tcoLoc[1]]
				matchEnd = matchStart + len(url)
			}
			result = append(result,
				&TwitterEntity{Text: url,
					ByteRange: Range{
						Start: matchStart + offset,
						Stop:  matchEnd + offset},
					Type: URL})
		}
	}

	// Add character/rune offsets in addition to byte offsets
	result.fixIndices(text)
	return result
}

// Extracts @username mentions from the supplied text. Returns a slice
// of TwitterEntity struct pointers.
//
// The ScreenName field in the returned structs will contain the value
// of the referenced username without the leading @ sign
func ExtractMentionedScreenNames(text string) []*TwitterEntity {
	mentionsOrLists := ExtractMentionsOrLists(text)
	var result []*TwitterEntity
	for _, e := range mentionsOrLists {
		if !e.listSlugIsSet {
			result = append(result, e)
		}
	}
	return result
}

// Extracts @username mentions or list names from the supplied text. Returns
// a slice of TwitterEntity struct pointers.
//
// The ScreenName field in the returned structs will contain the value of the
// referenced username without the leading @ sign.
//
// The ListSlug field in the returned structs will contain the name of the
// list (if present), without the leading / or preceding username
func ExtractMentionsOrLists(text string) []*TwitterEntity {
	// Optimization
	if !strings.ContainsAny(text, "@＠") {
		return nil
	}

	var result entitiesT
	matches := validMentionOrList.FindAllStringSubmatchIndex(text, -1)
	for _, m := range matches {
		matchEnd := text[m[1]:]
		if invalidMentionMatchEnd.MatchString(matchEnd) {
			continue
		}

		atSignStart := m[validMentionOrListGroupAt*2]
		screennameStart := m[validMentionOrListGroupUsername*2]
		screennameEnd := m[validMentionOrListGroupUsername*2+1]
		listNameStart := m[validMentionOrListGroupList*2]
		listNameEnd := m[validMentionOrListGroupList*2+1]

		var slug string
		start := atSignStart
		stop := screennameEnd
		if listNameStart > 0 {
			slug = text[listNameStart:listNameEnd]
			stop = listNameEnd
		}

		result = append(result, &TwitterEntity{
			Text:            text[start:stop],
			screenName:      text[screennameStart:screennameEnd],
			screenNameIsSet: true,
			listSlug:        slug,
			listSlugIsSet:   slug != "",
			ByteRange: Range{
				Start: start,
				Stop:  stop},
			Type: MENTION})
	}

	result.fixIndices(text)
	return result
}

// Extracts an @username mention from the beginning of the supplied text. A reply
// is defined as the occurrence of a mention (@username) at the beginning of a
// tweet preceded by 0 or more spaces.
//
// Returns a pointer to a TwitterEntity struct
func ExtractReplyScreenname(text string) *TwitterEntity {
	match := validReply.FindStringSubmatchIndex(text)
	if match != nil && !invalidMentionMatchEnd.MatchString(text[match[1]:]) {
		atSignStart := match[validReplyGroupAt*2]
		screennameStart := match[validReplyGroupUsername*2]
		screennameEnd := match[validReplyGroupUsername*2+1]
		entity := &TwitterEntity{
			Text:            text[atSignStart:screennameEnd],
			screenName:      text[screennameStart:screennameEnd],
			screenNameIsSet: true,
			ByteRange: Range{
				Start: atSignStart,
				Stop:  screennameEnd},
			Type: MENTION}

		start := utf8.RuneCountInString(text[:entity.ByteRange.Start])
		entity.Range.Start = start
		entity.Range.Stop = start + utf8.RuneCountInString(text[entity.ByteRange.Start:entity.ByteRange.Stop])
		return entity
	}
	return nil
}

// Extracts #hashtag occurrences from the supplied text. Returns a slice
// of TwitterEntity struct pointers.
//
// The Hashtag field of the returned entities will contain the value
// of the extracted hashtag without the leading # character
func ExtractHashtags(text string) []*TwitterEntity {
	return extractHashtags(text, true)
}

func extractHashtags(text string, checkUrlOverlap bool) []*TwitterEntity {
	// Optimization
	if !strings.ContainsAny(text, "#＃") {
		return nil
	}
	var result entitiesT
	var hashStart int
	var hashtagStart int
	var hashtagEnd int
	for _, match := range validHashtag.FindAllStringSubmatchIndex(text, -1) {
		if invalidHashtagMatchEnd.MatchString(text[match[1]:]) {
			continue
		}
		hashStart = match[validHashtagGroupHash*2]
		hashtagStart = match[validHashtagGroupTag*2]
		hashtagEnd = match[validHashtagGroupTag*2+1]
		result = append(result, &TwitterEntity{
			Text:         text[hashStart:hashtagEnd],
			hashtag:      text[hashtagStart:hashtagEnd],
			hashtagIsSet: true,
			ByteRange: Range{
				Start: hashStart,
				Stop:  hashtagEnd,
			},
			Type: HASH_TAG})
	}

	result.fixIndices(text)

	if checkUrlOverlap {
		urls := ExtractUrls(text)
		result = append(result, urls...)
		sort.Sort(result)
		result.removeOverlappingEntities()

		numHashtags := 0
		for _, e := range result {
			if e.Type == HASH_TAG {
				result[numHashtags] = e
				numHashtags++
			}
		}
		result = result[:numHashtags]

		var tmpResult []*TwitterEntity
		for _, e := range result {
			if e.Type == HASH_TAG {
				tmpResult = append(tmpResult, e)
			}
		}
		result = tmpResult
	}

	return result
}

// Extracts $cashtag occurrences from the supplied text. Returns a slice
// of TwitterEntity struct pointers.
//
// The Cashtag field of the returned entities will contain the value
// of the extracted cashtag without the leading $ character
func ExtractCashtags(text string) []*TwitterEntity {
	if !strings.Contains(text, "$") {
		return nil
	}

	var (
		result       entitiesT
		cashtagStart int
		cashtagEnd   int
		offset       int
		nextOffset   int
	)

	// Start at the beginning of the input string,
	// walking forward one match at a time. We have
	// to walk the string because the regexp package
	// lacks support for lookahead assertions
	for {
		offset = nextOffset
		substr := text[offset:]
		match := validCashtag.FindStringSubmatchIndex(substr)

		// If no matches are found in this portion of the string,
		// we're done
		if match == nil {
			break
		}

		cashtagStart = match[validCashtagGroupCashtag*2]
		cashtagEnd = match[validCashtagGroupCashtag*2+1]

		// Next time around, start at the end of the current match,
		// minus 1 because indices are not inclusive
		nextOffset = cashtagEnd + offset - 1

		result = append(result, &TwitterEntity{
			Text:         substr[cashtagStart-1 : cashtagEnd],
			cashtag:      substr[cashtagStart:cashtagEnd],
			cashtagIsSet: true,
			ByteRange: Range{
				Start: cashtagStart + offset - 1,
				Stop:  cashtagEnd + offset,
			},
			Type: CASH_TAG,
		})
	}
	result.fixIndices(text)
	return result
}
