package backend

import (
	"context"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	zuma "code.justin.tv/chat/zuma/client"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	usersservice "code.justin.tv/web/users-service/client"
	"github.com/cactus/go-statsd-client/statsd"
)

const (
	// FollowsKind is a follows relationship in cohesion
	FollowsKind = "follows"

	// UserKind is a user entity in cohesion
	UserKind = "user"

	// FollowersKind is a followed by relationship in cohesion
	FollowersKind = "followed_by"

	// HiddenFollowsKind is a hidden follow relationship
	HiddenFollowsKind = "hidden_follows"

	// HiddenFollowersKind is a hidden follower relationship
	HiddenFollowersKind = "hidden_followed_by"

	// SnsTimeout is the timeout for an attempted SNS message
	SnsTimeout = 3 * time.Second

	// SpadeTimeout is the timeout for an attempted Spade message
	SpadeTimeout = 3 * time.Second

	// PubSubTimeout is the timeout for an attempted PubSub message
	PubSubTimeout = 3 * time.Second

	// EventBusTimeout is the timeout for an attempted Event Bus publish
	EventBusTimeout = 3 * time.Second
)

// Backender contains functionality that all Backend objects must support
type Backender interface {
	CountFollows(ctx context.Context, userID string) (int, error)
	CountFollowers(ctx context.Context, userID string) (int, error)
	FollowUser(ctx context.Context, fromUserID string, targetUserID string, blockNotifications bool) error
	GetBatchFollows(ctx context.Context, fromUserID string, targetUserIDs []string) ([]*graphdbFulton.EdgeGetResponse, error)
	GetFollow(ctx context.Context, fromUserID string, targetUserID string) (*graphdbFulton.EdgeGetResponse, error)
	HideAllFollows(ctx context.Context, userID string) error
	ListFollowers(ctx context.Context, userID string, cursor string, limit int, offset int, direction string) (*graphdbFulton.EdgeListResponse, error)
	ListFollows(ctx context.Context, userID string, cursor string, limit int, offset int, direction string) (*graphdbFulton.EdgeListResponse, error)
	RestoreAllFollows(ctx context.Context, userID string) error
	UnfollowUser(ctx context.Context, fromUserID string, targetUserID string) error
	UpdateFollow(ctx context.Context, fromUserID string, targetUserID string, blockNotifications bool) error
	DestroyUser(ctx context.Context, userID string) error
	BulkUpdateFollows(ctx context.Context, userID string, limit int, edge, newEdge string) (bool, error)
	BulkDeleteFollows(ctx context.Context, userID string, limit int, edge string) (bool, error)
}

// Backend contains the primary writers and readers to various databases
type Backend struct {
	Client                   graphdb.GraphDB
	GraphDBFultonClient      graphdbFulton.TwitchVXGraphDBECS
	ErrorLogger              errorlogger.ErrorLogger
	Stats                    statsd.Statter
	SNS                      clients.SNSClient
	PubSub                   clients.PubSubClient
	UsersClient              usersservice.Client
	Spade                    clients.SpadeClient
	EventBus                 clients.EventBusClient
	ZumaClient               zuma.Client
	DartClient               clients.DartClient
	FollowBotDetectionClient clients.FollowBotDetectionClient
}

var _ Backender = Backend{}
