package backendcache

import (
	"bytes"
	"compress/gzip"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"sync"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/discovery/experiments"
	"code.justin.tv/discovery/experiments/experiment"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/foundation/gomemcache/memcache"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
)

// Cache allows storing memcache items
type Cache interface {
	Delete(ctx context.Context, key string) error
	Get(ctx context.Context, key string) (*memcache.Item, error)
	Set(ctx context.Context, item *memcache.Item) error
	IsDisabled() bool
	ExpirationTime() time.Duration
	ExpirationTimeLists() time.Duration
}

var cacheDirections = []string{
	"DESC", "ASC",
}

var cachedDirsMap = map[string]struct{}{
	"DESC": {},
	"ASC":  {},
}

const memcacheTimeout = 300 * time.Millisecond

var errMaxConcurrencyExceeded = errors.New("MaxDirtyConcurrency limit exceeded")

// CacheLimitationsConfig configures CacheLimitations
type CacheLimitationsConfig struct {
	allowedLimits *distconf.Str
}

// Load configuration from distconf
func (c *CacheLimitationsConfig) Load(d *distconf.Distconf) error {
	c.allowedLimits = d.Str("following-service.cache.allowed_limits", "25,100")
	return nil
}

// CacheLimitations put restrictions on what we can cache
type CacheLimitations struct {
	Config            *CacheLimitationsConfig
	mu                sync.RWMutex
	cachedLimitsSizes []int
	cachedLimitsMap   map[int]struct{}
}

// WatchCache calls reload when configuration changes
func (c *CacheLimitations) WatchCache() {
	c.Config.allowedLimits.Watch(func() {
		c.Reload()
	})
}

// Reload limit sizes from distconf
func (c *CacheLimitations) Reload() {
	sizes := c.Config.allowedLimits.Get()
	newCachedLimitsSizes := make([]int, len(sizes))
	newCachedLimitsMap := make(map[int]struct{}, len(sizes))
	for _, size := range strings.Split(sizes, ",") {
		sizeAsInt, err := strconv.Atoi(size)
		if err != nil {
			return
		}
		newCachedLimitsSizes = append(newCachedLimitsSizes, sizeAsInt)
		newCachedLimitsMap[sizeAsInt] = struct{}{}
	}
	c.mu.Lock()
	c.cachedLimitsSizes = newCachedLimitsSizes
	c.cachedLimitsMap = newCachedLimitsMap
	c.mu.Unlock()
}

// CanCacheLimit returns true if a limit can be cached
func (c *CacheLimitations) CanCacheLimit(limit int) bool {
	c.mu.RLock()
	_, canCache := c.cachedLimitsMap[limit]
	c.mu.RUnlock()
	return canCache
}

// CacheLimitSizes returns all the sizes we can cache.  Do not modify this list.
func (c *CacheLimitations) CacheLimitSizes() []int {
	c.mu.RLock()
	ret := c.cachedLimitsSizes
	c.mu.RUnlock()
	return ret
}

// BackendCache fronts a real backend with a cache for items
type BackendCache struct {
	Cache         Cache
	Statter       statsd.Statter
	CacheLimits   CacheLimitations
	DataBackend   backend.Backender
	ErrorLogger   errorlogger.ErrorLogger
	DirtyCh       chan string
	limitedLogger onceEvery
	Exps          experiments.Experiments
}

// A onceEvery calls the Do function at most once every Unit of time
type onceEvery struct {
	Unit     time.Duration
	mu       sync.Mutex
	nextTime time.Time
}

func (e *onceEvery) Do(f func()) {
	e.mu.Lock()
	defer e.mu.Unlock()
	now := time.Now()
	if now.After(e.nextTime) {
		e.nextTime = now.Add(e.unit())
		f()
	}
}

func (e *onceEvery) unit() time.Duration {
	if e.Unit == 0 {
		return time.Second
	}
	return e.Unit
}

var _ backend.Backender = &BackendCache{}

func (b *BackendCache) canCacheList(limit int, dir string) bool {
	s := b.CacheLimits.CanCacheLimit(limit)
	_, d := cachedDirsMap[strings.ToUpper(dir)]
	return s && d
}

func followsListKey(userID string, limit int, dir string) string {
	return "folkv2:" + userID + ":" + strings.ToUpper(dir) + ":" + strconv.Itoa(limit)
}

func followersListKey(userID string, limit int, dir string) string {
	return "frlkv2:" + userID + ":" + strings.ToUpper(dir) + ":" + strconv.Itoa(limit)
}

func followsCountKey(userID string) string {
	// FOllows Count Key. (fo-c-k)
	return "fock:" + userID
}

func followersCountKey(userID string) string {
	//FolloweRs Count Key. (fr-c-k)
	return "frck:" + userID
}

func (b *BackendCache) memcacheAnInt(ctx context.Context, hystrixKey string, statsdkey string, memcacheKey string, dataCallback func() (int, error)) (int, error) {
	var item *memcache.Item
	err := hystrix.Do(hystrixKey, func() (err error) {
		b.logError(b.Statter.Inc("backendv2cache."+statsdkey+".get", 1, .01))
		item, err = b.Cache.Get(ctx, memcacheKey)
		if err != nil {
			if err == memcache.ErrCacheMiss || item == nil {
				return nil
			}
			return err
		}
		return nil
	}, func(err error) error {
		b.ErrorLogger.Error(err)
		return nil
	})
	if err == nil && item != nil {
		b.logError(b.Statter.Inc("backendv2cache."+statsdkey+".hit", 1, .01))
		return strconv.Atoi(string(item.Value))
	}
	retInt, err := dataCallback()
	if err != nil {
		return 0, err
	}
	if cacheErr := b.setMemcacheItem(&memcache.Item{
		Key:        memcacheKey,
		Value:      []byte(strconv.Itoa(retInt)),
		Expiration: int32(b.Cache.ExpirationTime().Seconds()),
	}); cacheErr != nil {
		b.ErrorLogger.Error(cacheErr)
	}
	return retInt, err
}

// GZIP because these values can get large and memcache has a size limit
func decodeListResp(val []byte) (*graphdbFulton.EdgeListResponse, error) {
	var ret graphdbFulton.EdgeListResponse
	gz, err := gzip.NewReader(bytes.NewReader(val))
	if err != nil {
		return nil, err
	}
	if err := json.NewDecoder(gz).Decode(&ret); err != nil {
		return nil, err
	}
	if err := gz.Close(); err != nil {
		return nil, err
	}
	return &ret, nil
}

// GZIP because these values can get large and memcache has a size limit
func encodeListResp(response *graphdbFulton.EdgeListResponse) ([]byte, error) {
	out := bytes.Buffer{}
	gz := gzip.NewWriter(&out)
	if err := json.NewEncoder(gz).Encode(response); err != nil {
		return nil, err
	}
	if err := gz.Close(); err != nil {
		return nil, err
	}
	return out.Bytes(), nil
}

func (b *BackendCache) memcacheListResp(ctx context.Context, hystrixKey string, statsdkey string, memcacheKey string, dataCallback func() (*graphdbFulton.EdgeListResponse, error)) (*graphdbFulton.EdgeListResponse, error) {
	var item *memcache.Item
	err := hystrix.Do(hystrixKey, func() (err error) {
		b.logError(b.Statter.Inc("backendv2cache."+statsdkey+".get", 1, .01))
		item, err = b.Cache.Get(ctx, memcacheKey)
		if err != nil {
			if err == memcache.ErrCacheMiss || item == nil {
				return nil
			}
			return err
		}
		return nil
	}, func(err error) error {
		b.ErrorLogger.Error(err)
		return nil
	})
	if err == nil && item != nil {
		b.logError(b.Statter.Inc("backendv2cache."+statsdkey+".hit", 1, .01))
		return decodeListResp(item.Value)
	}
	retList, err := dataCallback()
	if err != nil {
		return nil, err
	}
	cacheVal, err := encodeListResp(retList)
	if err != nil {
		return nil, err
	}
	if cacheErr := b.setMemcacheItem(&memcache.Item{
		Key:        memcacheKey,
		Value:      cacheVal,
		Expiration: int32(b.Cache.ExpirationTimeLists().Seconds()),
	}); cacheErr != nil {
		b.ErrorLogger.Error(cacheErr)
	}
	return retList, err
}

func (b *BackendCache) setMemcacheItem(item *memcache.Item) error {
	ctx, cancel := context.WithTimeout(context.Background(), memcacheTimeout)
	defer cancel()
	return b.Cache.Set(ctx, item)
}

func (b *BackendCache) deleteKey(ctx context.Context, key string) {
	select {
	case b.DirtyCh <- key:
	default:
		b.limitedLogger.Do(func() {
			b.logError(errMaxConcurrencyExceeded)
		})
		b.logError(b.Cache.Delete(ctx, key))
	}
}

func (b *BackendCache) dirtyFollowerLists(ctx context.Context, userID string) {
	for _, cacheSize := range b.CacheLimits.CacheLimitSizes() {
		for _, cacheDir := range cacheDirections {
			b.deleteKey(ctx, followersListKey(userID, cacheSize, cacheDir))
		}
	}
}

func (b *BackendCache) dirtyFollowLists(ctx context.Context, userID string) {
	for _, cacheSize := range b.CacheLimits.CacheLimitSizes() {
		for _, cacheDir := range cacheDirections {
			b.deleteKey(ctx, followsListKey(userID, cacheSize, cacheDir))
		}
	}
}

// DirtyLoop runs forever till the main process is terminated
func (b *BackendCache) DirtyLoop() {
	for key := range b.DirtyCh {
		ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
		b.logError(b.Cache.Delete(ctx, key))
		cancel()
	}
}

// Monitor runs forever till the main process is terminated and sends metrics on the state of the cache
func (b *BackendCache) Monitor() {
	ticker := time.NewTicker(1 * time.Second)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			b.logError(b.Statter.Gauge("backendv2cache.dirty_channel.size", int64(len(b.DirtyCh)), 1))
		}
	}
}

func (b *BackendCache) logError(err error) {
	if err != nil && err != memcache.ErrCacheMiss {
		b.ErrorLogger.Error(err)
	}
}

func (b BackendCache) CountFollows(ctx context.Context, userID string) (int, error) {
	var count int
	var err error
	start := time.Now()
	if b.doRollout(clients.CacheRemovalExperimentName, clients.CacheRemovalExperimentID, userID) {
		count, err = b.DataBackend.CountFollows(ctx, userID)
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.countfollows.nocache", duration, 0.1))
	} else {
		count, err = b.memcacheAnInt(ctx, clients.HystrixGraphDBCacheCountFollows, "count_follows", followsCountKey(userID), func() (int, error) {
			return b.DataBackend.CountFollows(ctx, userID)
		})
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.countfollows.cache", duration, 0.1))
	}
	return count, err
}

func (b BackendCache) CountFollowers(ctx context.Context, userID string) (int, error) {
	var count int
	var err error
	start := time.Now()
	if b.doRollout(clients.CacheRemovalExperimentName, clients.CacheRemovalExperimentID, userID) {
		count, err = b.DataBackend.CountFollowers(ctx, userID)
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.countfollowers.nocache", duration, 0.1))
	} else {
		count, err = b.memcacheAnInt(ctx, clients.HystrixGraphDBCacheCountFollowers, "count_followers", followersCountKey(userID), func() (int, error) {
			return b.DataBackend.CountFollowers(ctx, userID)
		})
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.countfollowers.cache", duration, 0.1))
	}
	return count, err
}

func (b BackendCache) FollowUser(ctx context.Context, fromUserID string, targetUserID string, blockNotifications bool) error {
	b.clearCache(ctx, fromUserID, targetUserID)
	return b.DataBackend.FollowUser(ctx, fromUserID, targetUserID, blockNotifications)
}

func (b BackendCache) GetBatchFollows(ctx context.Context, fromUserID string, targetUserIDs []string) ([]*graphdbFulton.EdgeGetResponse, error) {
	return b.DataBackend.GetBatchFollows(ctx, fromUserID, targetUserIDs)
}

func (b BackendCache) GetFollow(ctx context.Context, fromUserID string, targetUserID string) (*graphdbFulton.EdgeGetResponse, error) {
	return b.DataBackend.GetFollow(ctx, fromUserID, targetUserID)
}

func (b BackendCache) dirtyAllFollows(userID string) {
	if b.Cache.IsDisabled() {
		return
	}
	hideCtx, cancel := context.WithTimeout(context.Background(), time.Second*20)
	defer cancel()
	gdbResp, err := b.ListFollows(hideCtx, userID, "", 2000, 0, "ASC")
	if err != nil {
		b.ErrorLogger.Error(err)
		return
	}
	for _, edge := range gdbResp.Edges {
		b.deleteKey(hideCtx, followersCountKey(edge.Edge.Edge.To.Id))
		b.dirtyFollowLists(hideCtx, edge.Edge.Edge.To.Id)
		b.dirtyFollowerLists(hideCtx, edge.Edge.Edge.To.Id)
	}
	b.dirtyFollowLists(hideCtx, userID)
	b.dirtyFollowerLists(hideCtx, userID)
}

func (b BackendCache) HideAllFollows(ctx context.Context, userID string) error {
	// Note: The count on the other direction isn't dirtied as an optimization for the common case.
	b.deleteKey(ctx, followsCountKey(userID))
	go b.dirtyAllFollows(userID)
	return b.DataBackend.HideAllFollows(ctx, userID)
}

func (b BackendCache) BulkUpdateFollows(ctx context.Context, userID string, limit int, edge, newEdge string) (bool, error) {
	return b.DataBackend.BulkUpdateFollows(ctx, userID, limit, edge, newEdge)
}

func (b BackendCache) BulkDeleteFollows(ctx context.Context, userID string, limit int, edge string) (bool, error) {
	return b.DataBackend.BulkDeleteFollows(ctx, userID, limit, edge)
}

func (b BackendCache) ListFollowers(ctx context.Context, userID string, cursor string, limit int, offset int, direction string) (*graphdbFulton.EdgeListResponse, error) {
	if cursor != "" || offset != 0 || !b.canCacheList(limit, direction) {
		return b.DataBackend.ListFollowers(ctx, userID, cursor, limit, offset, direction)
	}
	var resp *graphdbFulton.EdgeListResponse
	var err error
	start := time.Now()
	if b.doRollout(clients.CacheRemovalExperimentName, clients.CacheRemovalExperimentID, userID) {
		resp, err = b.DataBackend.ListFollowers(ctx, userID, cursor, limit, offset, direction)
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.listfollowers.nocache", duration, 0.1))
	} else {
		resp, err = b.memcacheListResp(ctx, clients.HystrixGraphDBCacheListFollowers, fmt.Sprintf("%s.%s.%d", "list_followers", direction, limit), followersListKey(userID, limit, direction), func() (*graphdbFulton.EdgeListResponse, error) {
			return b.DataBackend.ListFollowers(ctx, userID, cursor, limit, offset, direction)
		})
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.listfollowers.cache", duration, 0.1))
	}
	return resp, err
}

func (b BackendCache) ListFollows(ctx context.Context, userID string, cursor string, limit int, offset int, direction string) (*graphdbFulton.EdgeListResponse, error) {
	if cursor != "" || offset != 0 || !b.canCacheList(limit, direction) {
		return b.DataBackend.ListFollows(ctx, userID, cursor, limit, offset, direction)
	}
	var resp *graphdbFulton.EdgeListResponse
	var err error
	start := time.Now()
	if b.doRollout(clients.CacheRemovalExperimentName, clients.CacheRemovalExperimentID, userID) {
		resp, err = b.DataBackend.ListFollows(ctx, userID, cursor, limit, offset, direction)
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.listfollows.nocache", duration, 0.1))
	} else {
		resp, err = b.memcacheListResp(ctx, clients.HystrixGraphDBCacheListFollows, fmt.Sprintf("%s.%s.%d", "list_follows", direction, limit), followsListKey(userID, limit, direction), func() (*graphdbFulton.EdgeListResponse, error) {
			return b.DataBackend.ListFollows(ctx, userID, cursor, limit, offset, direction)
		})
		duration := time.Since(start)
		b.logError(b.Statter.TimingDuration("backendcache.listfollows.cache", duration, 0.1))
	}
	return resp, err
}

func (b BackendCache) RestoreAllFollows(ctx context.Context, userID string) error {
	b.deleteKey(ctx, followsCountKey(userID))
	b.deleteKey(ctx, followersCountKey(userID))
	b.dirtyFollowLists(ctx, userID)
	b.dirtyFollowerLists(ctx, userID)
	go b.dirtyAllFollows(userID)
	return b.DataBackend.RestoreAllFollows(ctx, userID)
}

func (b BackendCache) UnfollowUser(ctx context.Context, fromUserID string, targetUserID string) error {
	b.clearCache(ctx, fromUserID, targetUserID)
	return b.DataBackend.UnfollowUser(ctx, fromUserID, targetUserID)
}

func (b BackendCache) UpdateFollow(ctx context.Context, fromUserID string, targetUserID string, blockNotifications bool) error {
	b.clearCache(ctx, fromUserID, targetUserID)
	return b.DataBackend.UpdateFollow(ctx, fromUserID, targetUserID, blockNotifications)
}

func (b BackendCache) DestroyUser(ctx context.Context, userID string) error {
	b.dirtyAllFollows(userID)
	return b.DataBackend.DestroyUser(ctx, userID)
}

func (b BackendCache) clearCache(ctx context.Context, fromUserID string, targetUserID string) {
	b.deleteKey(ctx, followsCountKey(fromUserID))
	b.deleteKey(ctx, followersCountKey(targetUserID))
	b.dirtyFollowLists(ctx, fromUserID)
	b.dirtyFollowerLists(ctx, targetUserID)
}

func (b BackendCache) doRollout(expName string, expID string, userID string) bool {
	group, err := b.Exps.Treat(experiment.UserType, expName, expID, userID)
	if err != nil {
		return false
	}

	switch group {
	case "variant":
		return true
	default:
		return false
	}
}
