package backend_test

import (
	"errors"
	"testing"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/mocks"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type BatchOpSuite struct {
	suite.Suite
	backend backend.Backender
}

func (suite *BatchOpSuite) SetupSuite() {
	client := &mocks.TwitchVXGraphDBECS{}
	// successful update
	client.On("BulkEdgeUpdate", mock.Anything, &graphdbFulton.BulkEdgeUpdateRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "1",
		},
		EdgeType:    backend.FollowsKind,
		NewEdgeType: backend.HiddenFollowsKind,
		Limit:       10,
	}).Return(&graphdbFulton.BulkEdgeUpdateResponse{
		More: true,
	}, nil)
	// failed update
	client.On("BulkEdgeUpdate", mock.Anything, &graphdbFulton.BulkEdgeUpdateRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "err",
		},
		EdgeType:    backend.FollowsKind,
		NewEdgeType: backend.HiddenFollowsKind,
		Limit:       10,
	}).Return(nil, errors.New("BulkEdgeUpdate error"))
	// successful delete
	client.On("BulkEdgeDelete", mock.Anything, &graphdbFulton.BulkEdgeDeleteRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "1",
		},
		EdgeType: backend.FollowsKind,
		Limit:    10,
	}).Return(&graphdbFulton.BulkEdgeDeleteResponse{
		More: true,
	}, nil)
	// failed delete
	client.On("BulkEdgeDelete", mock.Anything, &graphdbFulton.BulkEdgeDeleteRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "1",
		},
		EdgeType: backend.FollowsKind,
		Limit:    10,
	}).Return(nil, errors.New("BulkEdgeDelete error"))
	suite.backend = backend.Backend{
		GraphDBFultonClient: client,
		ErrorLogger:         NewNoopErrorLogger(),
	}
}

func (suite *BatchOpSuite) TestSuccessfulUpdate() {
	actual, err := suite.backend.BulkUpdateFollows(nil, "1", 10, backend.FollowsKind, backend.HiddenFollowsKind)
	suite.Nil(err)
	suite.Equal(true, actual)
}

func (suite *BatchOpSuite) TestFailedUpdate() {
	_, err := suite.backend.BulkUpdateFollows(nil, "err", 10, backend.FollowsKind, backend.HiddenFollowsKind)
	suite.NotNil(err)
}

func (suite *BatchOpSuite) TestSuccessfulDelete() {
	actual, err := suite.backend.BulkDeleteFollows(nil, "1", 10, backend.FollowsKind)
	suite.Nil(err)
	suite.Equal(true, actual)
}

func (suite *BatchOpSuite) TestFailedDelete() {
	_, err := suite.backend.BulkDeleteFollows(nil, "err", 10, backend.FollowsKind)
	suite.NotNil(err)
}

func TestBatchOpSuite(t *testing.T) {
	suite.Run(t, new(BatchOpSuite))
}
