package backend

import (
	"context"
	"errors"
	"fmt"
	"log"
	"sync"

	followbotDetectionAPI "code.justin.tv/amzn/TwitchFollowBotDetectionTwirp"
	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	zumaAPI "code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/following-service/header"
	usersservice "code.justin.tv/web/users-service/client"
	"code.justin.tv/web/users-service/models"
	"github.com/afex/hystrix-go/hystrix"
)

// errFollowLimitExceeded is returned by FollowUser when a user already follows too many channels.
var errFollowLimitExceeded = errors.New("may not follow more than 2000 channels")

// errUserSuspended is returned by FollowUser when the "from user" is suspended
var errUserSuspended = errors.New("may not follow if suspended")

// errUserDeleted is returned by FollowUser when the "from user" is deleted
var errUserDeleted = errors.New("may not follow if deleted")

// errUserIsBlocked is returned by FollowUser when the "from user" is blocked
var errUserIsBlocked = errors.New("may not follow if blocked")

// errUserIsFollowBot is returned by FollowUser when the "from user" has been classified as a follow bot
var errUserIsFollowBot = errors.New("user is a follow bot")

func IsFollowLimitExceeded(err error) bool {
	return err != nil && err == errFollowLimitExceeded
}

func IsUserSuspended(err error) bool {
	return err != nil && err == errUserSuspended
}

func IsUserDeleted(err error) bool {
	return err != nil && err == errUserDeleted
}

func IsUserBlocked(err error) bool {
	return err != nil && err == errUserIsBlocked
}

func IsUserAFollowBot(err error) bool {
	return err != nil && err == errUserIsFollowBot
}

func (b Backend) FollowUser(ctx context.Context, fromUserID string, targetUserID string, blockNotifications bool) error {
	clientID := header.GetHeaderFromContext(ctx, header.ClientIDHeader)
	deviceID := header.GetHeaderFromContext(ctx, header.DeviceIDHeader)

	wg := sync.WaitGroup{}
	var errs [3]error
	wg.Add(3)
	go func() {
		defer wg.Done()
		errs[0] = b.validateUserNotBlocked(ctx, fromUserID, targetUserID)
	}()

	go func() {
		defer wg.Done()
		errs[1] = b.validateFollowCountUnderLimit(ctx, fromUserID)
	}()

	go func() {
		defer wg.Done()
		errs[2] = b.validateUserStatus(ctx, fromUserID)
	}()

	// this is purposely not added to the wait group because we want to run it in the background
	// without affecting latency or otherwise of regular traffic
	go func() {
		// using background context, todo switch to request ctx
		err := b.classifyFollowBotLikelihood(context.Background(), fromUserID, targetUserID, clientID, deviceID)
		if err != nil {
			// increase stat on errors
			err := b.Stats.Inc("followuser.followbot.yes", 1, 0.1)
			if err != nil {
				b.ErrorLogger.Error(err)
			}
		} else {
			err := b.Stats.Inc("followuser.followbot.no", 1, 0.1)
			if err != nil {
				b.ErrorLogger.Error(err)
			}
		}
	}()

	wg.Wait()
	for _, err := range errs {
		if err != nil {
			return err
		}
	}

	log.Printf("FollowUserv2 begin - fromUserID %v targetUserID %v \n", fromUserID, targetUserID)
	err := hystrix.Do(clients.HystrixGraphDBCreate, func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%s circuit panic=%v", clients.HystrixGraphDBCreate, p)
			}
		}()

		// check if recreating this will cause an error
		_, err = b.GraphDBFultonClient.EdgeCreate(ctx, &graphdbFulton.EdgeCreateRequest{
			Edge: &graphdbFulton.Edge{
				From: &graphdbFulton.Node{
					Type: UserKind,
					Id:   fromUserID,
				},
				To: &graphdbFulton.Node{
					Type: UserKind,
					Id:   targetUserID,
				},
				Type: FollowsKind,
			},
			Data: &graphdbFulton.DataBag{
				Bools: map[string]bool{
					"block_notifications": blockNotifications,
				},
			},
		})
		if err != nil {
			return err
		}

		return nil
	}, nil)
	log.Printf("FollowUserv2 end - fromUserID %v targetUserID %v err %v \n", fromUserID, targetUserID, err)

	if err != nil {
		return err
	}

	go b.sendFollowerCreatedEmail(targetUserID, fromUserID)
	go b.sendFollowMessages(targetUserID, fromUserID, clientID, deviceID)
	go b.sendSpadeNotificationStatusUpdateMessage(targetUserID, fromUserID, !blockNotifications, clientID, deviceID)

	return nil
}

// validateUserStatus returns if a user is in a suspended or deleted status.
func (b Backend) validateUserStatus(ctx context.Context, userID string) error {
	userProps, err := b.getUserByID(ctx, userID)
	if err != nil {
		if usersservice.IsUserNotFound(err) {
			return err
		}
		// fail open call to getUserByID
		b.ErrorLogger.Error(err)
		return nil
	}

	if userProps == nil || userProps.DeletedOn != nil {
		return errUserDeleted
	}

	if termsOfServiceViolation := userProps.TermsOfServiceViolation; termsOfServiceViolation != nil && *termsOfServiceViolation == true {
		return errUserSuspended

	}

	return nil
}

func (b Backend) validateUserNotBlocked(ctx context.Context, fromUserID string, targetUserID string) error {
	isBlockingResp, err := b.isBlocked(ctx, fromUserID, targetUserID)
	if err != nil {
		// If isBlocked call returns an error, log it and let the follow go through
		b.ErrorLogger.Error(err)
	} else if isBlockingResp.IsBlocked {
		return errUserIsBlocked
	}
	return nil
}

func (b Backend) validateFollowCountUnderLimit(ctx context.Context, fromUserID string) error {
	followLimitExceeded := false
	err := hystrix.Do(clients.HystrixGraphDBCountFollows, func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%s circuit panic=%v", clients.HystrixGraphDBCountFollows, p)
			}
		}()

		currentFollowCount, err := b.CountFollows(ctx, fromUserID)
		if err != nil {
			return err
		}
		if currentFollowCount >= 2000 {
			followLimitExceeded = true
		}
		return nil
	}, nil)
	if err != nil {
		return err
	}
	if followLimitExceeded {
		return errFollowLimitExceeded
	}
	return nil
}

func (b Backend) classifyFollowBotLikelihood(ctx context.Context, fromUserID string, targetUserID string, clientID string, deviceID string) error {
	classification, err := b.FollowBotDetectionClient.Classify(ctx, fromUserID, targetUserID, clientID, deviceID)
	if err != nil {
		// ignore error but log it for reference
		b.ErrorLogger.Error(err)
		return nil
	}

	// we will only drop if follow bot classification is "VERY LIKELY"
	if classification == int32(followbotDetectionAPI.Tier_VERY_LIKELY) {
		return errUserIsFollowBot
	}

	return nil
}

func (b Backend) sendFollowerCreatedEmail(targetUserID, fromUserID string) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	err := b.DartClient.PublishNotification(context.Background(), fromUserID, targetUserID)
	if err != nil {
		b.ErrorLogger.Error(err)
	}
}

func (b Backend) sendFollowMessages(targetUserID string, fromUserID string, clientID string, deviceID string) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	ctx := context.Background()

	followerCount, err := b.CountFollowers(ctx, targetUserID)
	followerCountPtr := &followerCount
	if err != nil {
		followerCountPtr = nil
		b.ErrorLogger.Error(err)
	}

	go b.sendSNSFollowMessage("follow", targetUserID, fromUserID, followerCountPtr)
	go b.sendSpadeFollowMessage(&clients.SpadeEventData{
		ClientID:     clientID,
		DeviceID:     deviceID,
		TargetUserID: targetUserID,
		FromUserID:   fromUserID,
		FollowCount:  followerCountPtr,
	})
	go b.sendPubSubFollowMessage(targetUserID, fromUserID)
	go b.sendEventBusFollowMessage(targetUserID, fromUserID, followerCount)
}

func (b Backend) sendSNSFollowMessage(eventType string, targetUserID string, fromUserID string, followerCount *int) {
	timeoutCtx, cancel := context.WithTimeout(context.Background(), SnsTimeout)
	defer cancel()

	err := b.SNS.SendUpdateFollowMessage(timeoutCtx, targetUserID, fromUserID, eventType, followerCount, b.ErrorLogger)
	if err != nil {
		b.ErrorLogger.Error(err)
	}
}

func (b Backend) sendPubSubFollowMessage(targetUserID, fromUserID string) {
	defer func() {
		if r := recover(); r != nil {
			b.ErrorLogger.Error(errorLogPanic(r))
		}
	}()

	ctx := context.Background()

	var fromUserProps *models.Properties
	fromUserProps, err := b.getUserByID(ctx, fromUserID)
	if err != nil {
		b.ErrorLogger.Error(err)
		return
	}

	var targetUserProps *models.Properties
	targetUserProps, err = b.getUserByID(ctx, targetUserID)
	if err != nil {
		b.ErrorLogger.Error(err)
		return
	}

	timeoutCtx, cancel := context.WithTimeout(ctx, PubSubTimeout)
	defer cancel()

	err = b.PubSub.SendFollowMessages(timeoutCtx, fromUserProps, targetUserProps, b.ErrorLogger)
	if err != nil {
		b.ErrorLogger.Error(err)
	}
}

func (b Backend) sendSpadeFollowMessage(event *clients.SpadeEventData) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	timeoutCtx, cancel := context.WithTimeout(context.Background(), SpadeTimeout)
	defer cancel()
	b.Spade.TrackFollowEvent(timeoutCtx, event)
}

func (b Backend) sendEventBusFollowMessage(targetUserID, fromUserID string, targetFollowerCount int) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	timeoutCtx, cancel := context.WithTimeout(context.Background(), EventBusTimeout)
	defer cancel()

	err := b.EventBus.SendFollowerCreatedMessage(timeoutCtx, targetUserID, fromUserID, targetFollowerCount, b.ErrorLogger)
	if err != nil {
		b.ErrorLogger.Error(err)
	}
}

func (b Backend) sendSpadeNotificationStatusUpdateMessage(targetUserID string, fromUserID string, allowsNotifications bool, clientID string, deviceID string) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	event := &clients.SpadeEventData{
		ClientID:            clientID,
		DeviceID:            deviceID,
		TargetUserID:        targetUserID,
		FromUserID:          fromUserID,
		AllowsNotifications: &allowsNotifications,
	}

	timeoutCtx, cancel := context.WithTimeout(context.Background(), SpadeTimeout)
	defer cancel()
	b.Spade.TrackNotificationStatusUpdateEvent(timeoutCtx, event)
}

func (b Backend) getUserByID(ctx context.Context, fromUserID string) (*models.Properties, error) {
	var fromUserProps *models.Properties
	err := hystrix.Do(clients.HystrixUsersServiceGetUserByID, func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%s circuit panic=%v", clients.HystrixUsersServiceGetUserByID, p)
			}
		}()

		fromUserProps, err = b.UsersClient.GetUserByID(ctx, fromUserID, nil)
		return err
	}, nil)
	return fromUserProps, err
}

func (b Backend) isBlocked(ctx context.Context, fromUserID string, targetUserID string) (zumaAPI.IsBlockedResponse, error) {
	var isBlockingResp zumaAPI.IsBlockedResponse
	err := hystrix.Do(clients.HystrixZumaIsBlocked, func() (err error) {
		// We check if the target user is blocking the from user
		isBlockingResp, err = b.ZumaClient.IsBlocked(ctx, zumaAPI.IsBlockedParams{
			UserID:        targetUserID,
			BlockedUserID: fromUserID,
		}, nil)
		return err
	}, nil)
	return isBlockingResp, err
}

func errorLogPanic(r interface{}) error {
	if e, ok := r.(error); ok {
		return e
	}
	return fmt.Errorf("%v", r)
}
