package backend_test

import (
	"context"
	"testing"
	"time"

	followbotDetectionAPI "code.justin.tv/amzn/TwitchFollowBotDetectionTwirp"
	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	zumaAPI "code.justin.tv/chat/zuma/app/api"
	zuma "code.justin.tv/chat/zuma/client"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/following-service/header"
	"code.justin.tv/feeds/following-service/mocks"
	"code.justin.tv/hygienic/errors"
	usersservice "code.justin.tv/web/users-service/client/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type FollowSuite struct {
	suite.Suite
	backend             backend.Backender
	graphdb             *mocks.GraphDB
	graphdbFultonClient *mocks.TwitchVXGraphDBECS
	sns                 *mocks.SNSClient
	pubsub              *mocks.PubSubClient
	spade               *mocks.SpadeClient
	users               *usersservice.Client
	zuma                *zuma.MockClient
	dart                *mocks.DartClient
	followbotdetection  *mocks.FollowBotDetectionClient
}

func (suite *FollowSuite) SetupTest() {
	client := &mocks.GraphDB{}
	graphDBFultonClient := &mocks.TwitchVXGraphDBECS{}
	sns := &mocks.SNSClient{}
	pubsub := &mocks.PubSubClient{}
	spade := &mocks.SpadeClient{}
	users := &usersservice.Client{}
	zumaMock := &zuma.MockClient{}
	dart := &mocks.DartClient{}
	followbotdetection := &mocks.FollowBotDetectionClient{}
	suite.backend = backend.Backend{
		Client:                   client,
		GraphDBFultonClient:      graphDBFultonClient,
		SNS:                      sns,
		PubSub:                   pubsub,
		Spade:                    spade,
		UsersClient:              users,
		ZumaClient:               zumaMock,
		DartClient:               dart,
		FollowBotDetectionClient: followbotdetection,
		ErrorLogger:              NewNoopErrorLogger(),
		Stats:                    &statsd.NoopClient{},
	}
	suite.sns = sns
	suite.pubsub = pubsub
	suite.graphdb = client
	suite.graphdbFultonClient = graphDBFultonClient
	suite.spade = spade
	suite.users = users
	suite.zuma = zumaMock
	suite.dart = dart
	suite.followbotdetection = followbotdetection
}

func (suite *FollowSuite) TestSuccessfulRequest() {
	suite.mockUsers(nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "2", true, nil)
	suite.mockSNSFollowUpdate("1", "2", nil)
	suite.mockSpade()
	suite.mockPubsub()
	suite.mockDart(nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)

	err := suite.backend.FollowUser(context.Background(), "1", "2", true)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdb.AssertExpectations(suite.T())
	suite.sns.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
	suite.dart.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestSuccessfulRequestWithHeadersPassed() {
	clientID := "client223"
	deviceID := "device123"
	blockNotifications := true
	allowsNotifications := !blockNotifications

	suite.mockUsers(nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "3", true, nil)
	suite.mockSNSFollowUpdate("1", "3", nil)
	suite.mockPubsub()
	suite.mockDart(nil)
	suite.mockFollowBotDetection("1", "3", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)
	suite.spade.On(
		"TrackFollowEvent",
		mock.Anything,
		&clients.SpadeEventData{
			ClientID:     clientID,
			DeviceID:     deviceID,
			FromUserID:   "1",
			TargetUserID: "3",
		},
	)
	suite.spade.On(
		"TrackNotificationStatusUpdateEvent",
		mock.Anything,
		&clients.SpadeEventData{
			ClientID:            clientID,
			DeviceID:            deviceID,
			FromUserID:          "1",
			TargetUserID:        "3",
			AllowsNotifications: &allowsNotifications,
		},
	)

	ctx := context.Background()
	ctx = context.WithValue(ctx, header.ClientIDHeader, clientID)
	ctx = context.WithValue(ctx, header.DeviceIDHeader, deviceID)

	err := suite.backend.FollowUser(ctx, "1", "3", true)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdb.AssertExpectations(suite.T())
	suite.sns.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
	suite.dart.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestFailedGraphDBCreate() {
	suite.mockUsers(nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "5", true, errors.New("Error"))
	suite.mockFollowBotDetection("1", "5", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)

	err := suite.backend.FollowUser(context.Background(), "1", "5", true)

	suite.Error(err)
}

func (suite *FollowSuite) TestSNSFailureDoesNotError() {
	suite.mockUsers(nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("10", 500, nil)
	suite.mockGraphDBCreate("10", "20", false, nil)
	suite.mockFollowBotDetection("10", "20", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)
	suite.mockSNSFollowUpdate("10", "20", errors.New("SNS error"))
	suite.mockSpade()
	suite.mockPubsub()
	suite.mockDart(nil)

	// a follow should not fail because a notification was not sent
	err := suite.backend.FollowUser(context.Background(), "10", "20", false)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdb.AssertExpectations(suite.T())
	suite.sns.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
	suite.dart.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestFollowLimitExceededError() {
	suite.mockUsers(nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("3", 2000, nil)
	suite.mockFollowBotDetection("3", "4", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)

	err := suite.backend.FollowUser(context.Background(), "3", "4", false)
	suite.True(backend.IsFollowLimitExceeded(err))
	suite.graphdb.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestUserErrorFailOpen() {
	suite.mockUsers(errors.New("super broken"))
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "2", true, nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)
	suite.mockSNSFollowUpdate("1", "2", nil)
	suite.mockSpade()
	suite.mockPubsub()
	suite.mockDart(nil)

	err := suite.backend.FollowUser(context.Background(), "1", "2", true)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdb.AssertExpectations(suite.T())
	suite.sns.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
	suite.dart.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestUserSuspended() {
	fromUserIsSuspended := true
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockZuma(false, nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)
	suite.users.On("GetUserByID", mock.Anything, mock.Anything, mock.Anything).Return(&models.Properties{TermsOfServiceViolation: &fromUserIsSuspended}, nil)

	err := suite.backend.FollowUser(context.Background(), "1", "2", false)
	suite.True(backend.IsUserSuspended(err))
	suite.graphdb.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestUserSuspensionNil() {
	suite.users.On("GetUserByID", mock.Anything, mock.Anything, mock.Anything).Return(&models.Properties{TermsOfServiceViolation: nil}, nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "2", true, nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)
	suite.mockSNSFollowUpdate("1", "2", nil)
	suite.mockSpade()
	suite.mockPubsub()
	suite.mockDart(nil)

	err := suite.backend.FollowUser(context.Background(), "1", "2", true)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdb.AssertExpectations(suite.T())
	suite.sns.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
	suite.dart.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestUserSuspensionFalse() {
	isSuspended := false
	suite.users.On("GetUserByID", mock.Anything, mock.Anything, mock.Anything).Return(&models.Properties{TermsOfServiceViolation: &isSuspended}, nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "2", true, nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)
	suite.mockSNSFollowUpdate("1", "2", nil)
	suite.mockSpade()
	suite.mockPubsub()
	suite.mockDart(nil)

	err := suite.backend.FollowUser(context.Background(), "1", "2", true)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdb.AssertExpectations(suite.T())
	suite.sns.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
	suite.dart.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestBlockedUserFollow() {
	suite.mockUsers(nil)
	suite.mockGraphDBCount("1", 0, nil)
	suite.mockZuma(true, nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_VERY_UNLIKELY, nil)

	err := suite.backend.FollowUser(context.Background(), "1", "2", false)
	suite.True(backend.IsUserBlocked(err))
	suite.graphdb.AssertExpectations(suite.T())
}

func (suite *FollowSuite) TestFollowBotDetectionFailure() {
	suite.mockUsers(nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "2", true, nil)
	suite.mockSNSFollowUpdate("1", "2", nil)
	suite.mockSpade()
	suite.mockPubsub()
	suite.mockDart(nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_UNKNOWN, errors.New("500"))

	err := suite.backend.FollowUser(context.Background(), "1", "2", true)
	suite.NoError(err, "should silently fail any follow bot detection service failures")
}

func (suite *FollowSuite) TestFollowBotDetectionLikelyFollowBot() {
	suite.mockUsers(nil)
	suite.mockZuma(false, nil)
	suite.mockGraphDBCount("1", 500, nil)
	suite.mockGraphDBCreate("1", "2", true, nil)
	suite.mockSNSFollowUpdate("1", "2", nil)
	suite.mockSpade()
	suite.mockPubsub()
	suite.mockDart(nil)
	suite.mockFollowBotDetection("1", "2", followbotDetectionAPI.Tier_VERY_LIKELY, nil)

	err := suite.backend.FollowUser(context.Background(), "1", "2", true)
	suite.NoError(err)
	// todo after darklaunch, uncomment the following assertions and remove the above line
	// we're silent failing so no errors are returned right now
	//suite.Error(err)
	//suite.True(backend.IsUserAFollowBot(err))
}

func (suite *FollowSuite) mockGraphDBCreate(fromUserID string, targetUserID string, blockNotifications bool, err error) {
	suite.graphdbFultonClient.On("EdgeCreate", mock.Anything, &graphdbFulton.EdgeCreateRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   fromUserID,
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   targetUserID,
			},
			Type: backend.FollowsKind,
		},
		Data: &graphdbFulton.DataBag{
			Bools: map[string]bool{
				"block_notifications": blockNotifications,
			},
		},
	}).Return(&graphdbFulton.EdgeCreateResponse{}, err)
}

func (suite *FollowSuite) mockGraphDBCount(userID string, count int64, err error) {
	suite.graphdbFultonClient.On("EdgeCount", mock.Anything, &graphdbFulton.EdgeCountRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   userID,
		},
		EdgeType: backend.FollowsKind,
	}).Return(&graphdbFulton.EdgeCountResponse{
		Count: count,
	}, err)
}

func (suite *FollowSuite) mockUsers(err error) {
	suite.users.On("GetUserByID",
		mock.Anything,
		mock.Anything,
		mock.Anything).Return(&models.Properties{}, err)
}

func (suite *FollowSuite) mockZuma(isBlocked bool, err error) {
	suite.zuma.On(
		"IsBlocked",
		mock.Anything,
		mock.Anything,
		mock.Anything).Return(zumaAPI.IsBlockedResponse{IsBlocked: isBlocked}, err)
}

func (suite *FollowSuite) mockSNSFollowUpdate(fromUserID string, targetUserID string, err error) {
	suite.sns.On("SendUpdateFollowMessage",
		mock.Anything,
		targetUserID,
		fromUserID,
		"follow",
		mock.Anything,
		mock.Anything).Return(err)
}

func (suite *FollowSuite) mockSpade() {
	suite.spade.On("TrackFollowEvent",
		mock.Anything,
		mock.Anything).Return()
	suite.spade.On("TrackNotificationStatusUpdateEvent",
		mock.Anything,
		mock.Anything).Return()
}

func (suite *FollowSuite) mockPubsub() {
	suite.pubsub.On("SendFollowMessages", mock.Anything, mock.Anything, mock.Anything, mock.Anything)
}

func (suite *FollowSuite) mockDart(err error) {
	suite.dart.On(
		"PublishNotification",
		context.Background(),
		mock.Anything,
		mock.Anything,
	).Return(err)
}

func (suite *FollowSuite) mockFollowBotDetection(userID string, targetUserID string, tier followbotDetectionAPI.Tier, err error) {
	suite.followbotdetection.On(
		"Classify",
		mock.Anything,
		userID,
		targetUserID,
		mock.Anything,
		mock.Anything,
	).Return(int32(tier), err)
}

func TestFollowSuite(t *testing.T) {
	suite.Run(t, new(FollowSuite))
}
