package backend

import (
	"context"
	"fmt"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/following-service/header"
	"github.com/afex/hystrix-go/hystrix"
)

const (
	// https://twitch.slack.com/archives/C01B3HSCDEZ/p1600974953003700?thread_ts=1600974181.002900&cid=C01B3HSCDEZ
	IOS_IPHONE_CLIENT_ID = "85lcqzxpb9bqu9z6ga1ol55du"

	IOS_IPAD_CLIENT_ID = "p9lhq6azjkdl72hs5xnt3amqu7vv8k2"

	DUMMY_CURSOR = "END"
)

func (b Backend) ListFollows(ctx context.Context, userID string, cursor string, limit int, offset int, direction string) (*graphdbFulton.EdgeListResponse, error) {
	// IOS hack for https://jira.twitch.com/browse/IOS-13560
	// We know that we will only be getting a DUMMY_CURSOR from mobile and they want an empty result
	// We can save the network call to our dependency and exit early
	clientID := header.GetHeaderFromContext(ctx, header.ClientIDHeader)
	if isIOS(clientID) && cursor == DUMMY_CURSOR {
		return &graphdbFulton.EdgeListResponse{}, nil
	}
	// END

	var response *graphdbFulton.EdgeListResponse
	err := hystrix.Do(clients.HystrixGraphDBListFollows, func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%s circuit panic=%v", clients.HystrixGraphDBListFollows, p)
			}
		}()

		order := graphdbFulton.PagedRequest_ASC
		if direction == "desc" {
			order = graphdbFulton.PagedRequest_DESC
		}

		response, err = b.GraphDBFultonClient.EdgeList(ctx, &graphdbFulton.EdgeListRequest{
			From: &graphdbFulton.Node{
				Type: UserKind,
				Id:   userID,
			},
			EdgeType: FollowsKind,
			Page: &graphdbFulton.PagedRequest{
				Cursor: cursor,
				Limit:  int64(limit),
				Order:  order,
			},
		})
		return err
	}, nil)
	// IOS hack
	if err == nil && isIOS(clientID) {
		if len(response.Edges) != 0 && response.Edges[len(response.Edges)-1].Cursor == "" {
			response.Edges[len(response.Edges)-1].Cursor = DUMMY_CURSOR
		}
	}
	// END
	return response, err
}

func isIOS(clientID string) bool {
	return (clientID == IOS_IPAD_CLIENT_ID) || (clientID == IOS_IPHONE_CLIENT_ID)
}
