package backend_test

import (
	"context"
	"errors"
	"testing"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/header"
	"code.justin.tv/feeds/following-service/mocks"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type ListFollowsSuite struct {
	suite.Suite
	backend       backend.Backender
	graphdbFulton *mocks.TwitchVXGraphDBECS
}

func (suite *ListFollowsSuite) SetupTest() {
	client := &mocks.TwitchVXGraphDBECS{}
	suite.backend = backend.Backend{
		GraphDBFultonClient: client,
		ErrorLogger:         NewNoopErrorLogger(),
	}
	suite.graphdbFulton = client
}

func (suite *ListFollowsSuite) TestSuccessfulRequest() {
	suite.graphdbFulton.On("EdgeList", mock.Anything, &graphdbFulton.EdgeListRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "1",
		},
		EdgeType: backend.FollowsKind,
		Page: &graphdbFulton.PagedRequest{
			Limit: 20,
			Order: graphdbFulton.PagedRequest_ASC,
		},
	}).Return(&graphdbFulton.EdgeListResponse{}, nil)
	follows, err := suite.backend.ListFollows(context.Background(), "1", "", 20, 0, "asc")
	suite.Nil(err)
	suite.NotNil(follows)
}

func (suite *ListFollowsSuite) TestFailedRequest() {
	suite.graphdbFulton.On("EdgeList", mock.Anything, &graphdbFulton.EdgeListRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "2",
		},
		EdgeType: backend.FollowsKind,
		Page: &graphdbFulton.PagedRequest{
			Limit: 20,
			Order: graphdbFulton.PagedRequest_ASC,
		},
	}).Return(nil, errors.New("ErrorError"))
	follows, err := suite.backend.ListFollows(context.Background(), "2", "", 20, 0, "asc")
	suite.EqualError(err, "ErrorError")
	suite.Nil(follows)
}

func (suite *ListFollowsSuite) TestPanic() {
	suite.graphdbFulton.On("EdgeList", mock.Anything, &graphdbFulton.EdgeListRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "3",
		},
		EdgeType: backend.FollowsKind,
		Page: &graphdbFulton.PagedRequest{
			Limit: 20,
			Order: graphdbFulton.PagedRequest_ASC,
		},
	}).Return(nil, nil).Run(func(arg mock.Arguments) { panic("panic") })
	follows, err := suite.backend.ListFollows(context.Background(), "3", "", 20, 0, "asc")
	suite.EqualError(err, "graphdb_listfollows circuit panic=panic")
	suite.Nil(follows)
}

func (suite *ListFollowsSuite) TestIOSDummyCursorWithHeader() {
	clientID := backend.IOS_IPAD_CLIENT_ID
	ctx := context.Background()
	ctx = context.WithValue(ctx, header.ClientIDHeader, clientID)

	suite.graphdbFulton.On("EdgeList", mock.Anything, &graphdbFulton.EdgeListRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "1",
		},
		EdgeType: backend.FollowsKind,
		Page: &graphdbFulton.PagedRequest{
			Limit: 20,
			Order: graphdbFulton.PagedRequest_ASC,
		},
	}).Return(&graphdbFulton.EdgeListResponse{
		Edges: []*graphdbFulton.LoadedCursoredEdge{
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "1",
						},
						Type: backend.FollowsKind,
					},
				},
				Cursor: "123456",
			},
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "1",
						},
						Type: backend.FollowsKind,
					},
				},
			},
		},
	}, nil)

	follows, err := suite.backend.ListFollows(ctx, "1", "", 20, 0, "asc")
	suite.Nil(err)
	suite.NotNil(follows)

	suite.Assert().Equal(2, len(follows.Edges), "length of response should not change")
	suite.Assert().Equal(backend.DUMMY_CURSOR, follows.Edges[len(follows.Edges)-1].Cursor)
}

func (suite *ListFollowsSuite) TestIOSDummyCursorWithOutHeader() {
	clientID := "NOTIOS"
	ctx := context.Background()
	ctx = context.WithValue(ctx, header.ClientIDHeader, clientID)

	suite.graphdbFulton.On("EdgeList", mock.Anything, &graphdbFulton.EdgeListRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "1",
		},
		EdgeType: backend.FollowsKind,
		Page: &graphdbFulton.PagedRequest{
			Limit: 20,
			Order: graphdbFulton.PagedRequest_ASC,
		},
	}).Return(&graphdbFulton.EdgeListResponse{
		Edges: []*graphdbFulton.LoadedCursoredEdge{
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "1",
						},
						Type: backend.FollowsKind,
					},
				},
				Cursor: "123456",
			},
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "1",
						},
						Type: backend.FollowsKind,
					},
				},
			},
		},
	}, nil)

	follows, err := suite.backend.ListFollows(ctx, "1", "", 20, 0, "asc")
	suite.Nil(err)
	suite.NotNil(follows)

	suite.Assert().Equal(2, len(follows.Edges), "length of response should not change")
	suite.Assert().Equal("", follows.Edges[len(follows.Edges)-1].Cursor)
}

func (suite *ListFollowsSuite) TestIOSDummyCursor() {
	clientID := backend.IOS_IPHONE_CLIENT_ID
	ctx := context.Background()
	ctx = context.WithValue(ctx, header.ClientIDHeader, clientID)

	follows, err := suite.backend.ListFollows(ctx, "1", backend.DUMMY_CURSOR, 20, 0, "asc")
	suite.Nil(err)
	suite.Assert().Equal(0, len(follows.Edges))
}

func TestListFollowsSuite(t *testing.T) {
	suite.Run(t, new(ListFollowsSuite))
}
