package backend

import (
	"context"
	"fmt"
	"log"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/following-service/header"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/pkg/errors"
)

func (b Backend) UnfollowUser(ctx context.Context, fromUserID string, targetUserID string) error {
	log.Printf("UnfollowUserv2 begin - fromUserID %v targetUserID %v \n", fromUserID, targetUserID)
	err := hystrix.Do(clients.HystrixGraphDBDelete, func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%s circuit panic=%v", clients.HystrixGraphDBDelete, p)
			}
		}()

		_, err = b.GraphDBFultonClient.EdgeDelete(ctx, &graphdbFulton.EdgeDeleteRequest{
			Edge: &graphdbFulton.Edge{
				From: &graphdbFulton.Node{
					Type: UserKind,
					Id:   fromUserID,
				},
				To: &graphdbFulton.Node{
					Type: UserKind,
					Id:   targetUserID,
				},
				Type: FollowsKind,
			},
		})
		if err != nil {
			return err
		}

		return nil
	}, nil)
	log.Printf("UnfollowUser end - fromUserID %v targetUserID %v err %v \n", fromUserID, targetUserID, err)
	if err != nil {
		return err
	}

	clientID := header.GetHeaderFromContext(ctx, header.ClientIDHeader)
	deviceID := header.GetHeaderFromContext(ctx, header.DeviceIDHeader)

	go b.sendUnfollowUpdateMessages(targetUserID, fromUserID, clientID, deviceID)

	return nil
}

func (b Backend) sendUnfollowUpdateMessages(targetUserID string, fromUserID string, clientID string, deviceID string) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	ctx := context.Background()

	followerCount, err := b.CountFollowers(ctx, targetUserID)
	followerCountPtr := &followerCount
	if err != nil {
		followerCountPtr = nil
		b.ErrorLogger.Error(err)
	}

	go b.sendSNSFollowMessage("unfollow", targetUserID, fromUserID, followerCountPtr)
	go b.sendSpadeUnfollowMessage(&clients.SpadeEventData{
		ClientID:     clientID,
		DeviceID:     deviceID,
		TargetUserID: targetUserID,
		FromUserID:   fromUserID,
		FollowCount:  followerCountPtr,
	})
	go func() {
		if err := b.PubSub.SendUnfollowMessageToFollower(ctx, targetUserID, fromUserID, b.ErrorLogger); err != nil {
			b.ErrorLogger.Error(errors.Wrap(err, "sending unfollow message to follower"))
		}
	}()
	go b.sendEventBusUnfollowMessage(targetUserID, fromUserID, followerCount)
}

func (b Backend) sendSpadeUnfollowMessage(event *clients.SpadeEventData) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	timeoutCtx, cancel := context.WithTimeout(context.Background(), SpadeTimeout)
	defer cancel()
	b.Spade.TrackUnfollowEvent(timeoutCtx, event)
}

func (b Backend) sendEventBusUnfollowMessage(targetUserID, fromUserID string, targetFollowerCount int) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("panic: %+v\n", r)
		}
	}()

	timeoutCtx, cancel := context.WithTimeout(context.Background(), EventBusTimeout)
	defer cancel()

	err := b.EventBus.SendFollowerDeletedMessage(timeoutCtx, targetUserID, fromUserID, targetFollowerCount, b.ErrorLogger)
	if err != nil {
		b.ErrorLogger.Error(err)
	}
}
