package backend_test

import (
	"context"
	"errors"
	"testing"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/following-service/header"
	"code.justin.tv/feeds/following-service/mocks"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type UnfollowSuite struct {
	suite.Suite
	backend backend.Backender
	sns     *mocks.SNSClient
	spade   *mocks.SpadeClient
	pubsub  *mocks.PubSubClient
}

func (suite *UnfollowSuite) SetupTest() {
	client := &mocks.TwitchVXGraphDBECS{}
	client.On("EdgeDelete", mock.Anything, &graphdbFulton.EdgeDeleteRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "1",
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "2",
			},
			Type: backend.FollowsKind,
		},
	}).Return(nil, nil)
	client.On("EdgeDelete", mock.Anything, &graphdbFulton.EdgeDeleteRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "2",
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "23",
			},
			Type: backend.FollowsKind,
		},
	}).Return(nil, nil)
	client.On("EdgeDelete", mock.Anything, &graphdbFulton.EdgeDeleteRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "1",
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "3",
			},
			Type: backend.FollowsKind,
		},
	}).Return(nil, errors.New("Error"))
	client.On("EdgeDelete", mock.Anything, &graphdbFulton.EdgeDeleteRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "1",
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "4",
			},
			Type: backend.FollowsKind,
		},
	}).Return(nil, nil).Run(func(arg mock.Arguments) { panic("panic") })
	client.On("EdgeCount", mock.Anything, &graphdbFulton.EdgeCountRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "1",
		},
		EdgeType: backend.FollowsKind,
	}).Return(&graphdbFulton.EdgeCountResponse{
		Count: 0,
	}, nil)
	client.On("EdgeCount", mock.Anything, &graphdbFulton.EdgeCountRequest{
		From: &graphdbFulton.Node{
			Type: backend.UserKind,
			Id:   "2",
		},
		EdgeType: backend.FollowsKind,
	}).Return(&graphdbFulton.EdgeCountResponse{
		Count: 0,
	}, nil)

	suite.sns = &mocks.SNSClient{}
	suite.spade = &mocks.SpadeClient{}
	suite.pubsub = &mocks.PubSubClient{}

	suite.backend = backend.Backend{
		GraphDBFultonClient: client,
		SNS:                 suite.sns,
		Spade:               suite.spade,
		PubSub:              suite.pubsub,
	}
}

func (suite *UnfollowSuite) TestSuccessfulRequest() {
	suite.sns.On("SendUpdateFollowMessage", mock.Anything, "2", "1", "unfollow", mock.Anything, mock.Anything).Return(nil)
	suite.spade.On("TrackUnfollowEvent", mock.Anything, mock.Anything).Return()
	suite.pubsub.On("SendUnfollowMessageToFollower", mock.Anything, "2", "1", mock.Anything)

	err := suite.backend.UnfollowUser(context.Background(), "1", "2")
	suite.Nil(err)
}

func (suite *UnfollowSuite) TestSuccessfulRequestWithHeadersPassed() {
	followCount := 0
	clientID := "client123"
	deviceID := "device1"
	suite.sns.On("SendUpdateFollowMessage", mock.Anything, "23", "2", "unfollow", mock.Anything, mock.Anything).Return(nil)
	suite.spade.On("TrackUnfollowEvent", mock.Anything, &clients.SpadeEventData{
		ClientID:     clientID,
		DeviceID:     deviceID,
		FromUserID:   "2",
		TargetUserID: "23",
		FollowCount:  &followCount,
	}).Return()
	suite.pubsub.On("SendUnfollowMessageToFollower", mock.Anything, "23", "2", mock.Anything)
	ctx := context.Background()
	ctx = context.WithValue(ctx, header.ClientIDHeader, clientID)
	ctx = context.WithValue(ctx, header.DeviceIDHeader, deviceID)

	err := suite.backend.UnfollowUser(ctx, "2", "23")
	time.Sleep(100 * time.Millisecond) // spade sends on a goroutine so need to wait

	suite.Nil(err)
}

func (suite *UnfollowSuite) TestFailedRequest() {
	err := suite.backend.UnfollowUser(context.Background(), "1", "3")
	suite.NotNil(err)
	suite.EqualError(err, "Error")
}

func (suite *UnfollowSuite) TestPanic() {
	err := suite.backend.UnfollowUser(context.Background(), "1", "4")
	suite.NotNil(err)
	suite.EqualError(err, "graphdb_delete circuit panic=panic")
}

func TestUnfollowSuite(t *testing.T) {
	suite.Run(t, new(UnfollowSuite))
}
