package backend_test

import (
	"context"
	"errors"
	"testing"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/following-service/header"
	"code.justin.tv/feeds/following-service/mocks"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type UpdateFollowSuite struct {
	suite.Suite
	backend       backend.Backender
	graphdbFulton *mocks.TwitchVXGraphDBECS
	spade         *mocks.SpadeClient
}

func (suite *UpdateFollowSuite) SetupTest() {
	client := &mocks.TwitchVXGraphDBECS{}
	spadeMock := &mocks.SpadeClient{}

	suite.backend = backend.Backend{
		GraphDBFultonClient: client,
		Spade:               spadeMock,
		ErrorLogger:         NewNoopErrorLogger(),
	}
	suite.graphdbFulton = client
	suite.spade = spadeMock
}

func (suite *UpdateFollowSuite) TestSuccessfulRequest() {
	allowsNotifications := false
	suite.graphdbFulton.On("EdgeUpdate", mock.Anything, &graphdbFulton.EdgeUpdateRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "1",
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "2",
			},
			Type: backend.FollowsKind,
		},
		Data: &graphdbFulton.DataBag{
			Bools: map[string]bool{
				"block_notifications": !allowsNotifications,
			},
		},
	}).Return(nil, nil)
	suite.mockSpade("1", "2", allowsNotifications)

	err := suite.backend.UpdateFollow(context.Background(), "1", "2", !allowsNotifications)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdbFulton.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
}

func (suite *UpdateFollowSuite) TestSuccessfulRequestWithHeadersPassed() {
	allowsNotifications := false
	clientID := "client123"
	deviceID := "device123"
	suite.spade.On("TrackNotificationStatusUpdateEvent", mock.Anything, &clients.SpadeEventData{
		FromUserID:          "3",
		TargetUserID:        "4",
		AllowsNotifications: &allowsNotifications,
		ClientID:            clientID,
		DeviceID:            deviceID,
	})
	suite.graphdbFulton.On("EdgeUpdate", mock.Anything, &graphdbFulton.EdgeUpdateRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "3",
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "4",
			},
			Type: backend.FollowsKind,
		},
		Data: &graphdbFulton.DataBag{
			Bools: map[string]bool{
				"block_notifications": !allowsNotifications,
			},
		},
	}).Return(nil, nil)

	ctx := context.Background()
	ctx = context.WithValue(ctx, header.ClientIDHeader, clientID)
	ctx = context.WithValue(ctx, header.DeviceIDHeader, deviceID)

	err := suite.backend.UpdateFollow(ctx, "3", "4", !allowsNotifications)
	time.Sleep(200 * time.Millisecond)

	suite.NoError(err)
	suite.graphdbFulton.AssertExpectations(suite.T())
	suite.spade.AssertExpectations(suite.T())
}

func (suite *UpdateFollowSuite) TestFailedRequest() {
	suite.graphdbFulton.On("EdgeUpdate", mock.Anything, &graphdbFulton.EdgeUpdateRequest{
		Edge: &graphdbFulton.Edge{
			From: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "5",
			},
			To: &graphdbFulton.Node{
				Type: backend.UserKind,
				Id:   "6",
			},
			Type: backend.FollowsKind,
		},
		Data: &graphdbFulton.DataBag{
			Bools: map[string]bool{
				"block_notifications": true,
			},
		},
	}).Return(nil, errors.New("broken"))

	err := suite.backend.UpdateFollow(context.Background(), "5", "6", true)

	suite.NotNil(err)
	suite.Error(err, "broken")
	suite.graphdbFulton.AssertExpectations(suite.T())
}

func (suite *UpdateFollowSuite) mockSpade(fromUserID string, targetUserID string, allowsNotifications bool) {
	suite.spade.On("TrackNotificationStatusUpdateEvent", mock.Anything, &clients.SpadeEventData{
		FromUserID:          fromUserID,
		TargetUserID:        targetUserID,
		AllowsNotifications: &allowsNotifications,
	})
}

func TestUpdateFollowSuite(t *testing.T) {
	suite.Run(t, new(UpdateFollowSuite))
}
