package cache

import (
	"time"

	"context"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/foundation/gomemcache/memcache"
)

// Config configures the cache
type Config struct {
	DisableCache        *distconf.Bool
	ExpirationTime      *distconf.Duration
	ExpirationTimeLists *distconf.Duration
}

// Load configuration from distconf
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.DisableCache = dconf.Bool("following-service.cache.disabled", false)
	c.ExpirationTime = dconf.Duration("following-service.cache.expiration_time", time.Hour*24)
	c.ExpirationTimeLists = dconf.Duration("following-service.cache.expiration_time_lists", time.Hour*6)
	return nil
}

// MemcacheCache stores items with memcache
type MemcacheCache struct {
	MemcacheClient *memcache.Client
	Config         *Config
}

// Delete removes an item from the cache
func (m *MemcacheCache) Delete(ctx context.Context, key string) error {
	if m.Config.DisableCache.Get() {
		return nil
	}
	return m.MemcacheClient.Delete(ctx, key)
}

// Get an item from the cache
func (m *MemcacheCache) Get(ctx context.Context, key string) (*memcache.Item, error) {
	if m.Config.DisableCache.Get() {
		return nil, nil
	}
	return m.MemcacheClient.Get(ctx, key)
}

// Set an item into the cache
func (m *MemcacheCache) Set(ctx context.Context, item *memcache.Item) error {
	if m.Config.DisableCache.Get() {
		return nil
	}
	return m.MemcacheClient.Set(ctx, item)
}

// IsDisabled returns if the cache is disabled
func (m *MemcacheCache) IsDisabled() bool {
	return m.Config.DisableCache.Get()
}

// ExpirationTime returns the default expiration time
func (m *MemcacheCache) ExpirationTime() time.Duration {
	return m.Config.ExpirationTime.Get()
}

// ExpirationTimeLists returns the expiration time for lists
func (m *MemcacheCache) ExpirationTimeLists() time.Duration {
	return m.Config.ExpirationTimeLists.Get()
}
