// +build integration

package follows_test

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"testing"
	"time"

	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/smartystreets/goconvey/convey"
)

func loadEnv(env string) (twitchclient.ClientConf, error) {
	f, err := os.Open("integration_test_endpoints.json")
	if err != nil {
		return twitchclient.ClientConf{}, err
	}
	defer func() {
		if err := f.Close(); err != nil {
			panic(err)
		}
	}()
	c := make(map[string]twitchclient.ClientConf)
	if err := json.NewDecoder(f).Decode(&c); err != nil {
		return twitchclient.ClientConf{}, err
	}
	if _, exists := c[env]; !exists {
		return twitchclient.ClientConf{}, fmt.Errorf("unable to find env %s in endpoints file", env)
	}
	return c[env], nil
}

func userIsEmpty(c C, ctx context.Context, client followsrpc.Follows, userID string) {
	res, err := client.CountFollowers(ctx, &followsrpc.UserIDReq{
		UserId: userID,
	})
	c.So(err, ShouldBeNil)
	c.So(res.Count, ShouldEqual, 0)

	res2, err := client.ListFollowers(ctx, &followsrpc.FollowsReq{
		UserId: userID,
	})
	c.So(err, ShouldBeNil)
	c.So(0, ShouldEqual, len(res2.Follows))

	res3, err := client.ListFollows(ctx, &followsrpc.FollowsReq{
		UserId: userID,
	})
	c.So(err, ShouldBeNil)
	c.So(0, ShouldEqual, len(res3.Follows))
}

func shouldNotFollow(c C, ctx context.Context, client followsrpc.Follows, fromUserID, toUserID string) {
	_, err := client.GetFollow(ctx, &followsrpc.FollowReq{
		FromUserId:   fromUserID,
		TargetUserId: toUserID,
	})
	c.So(err, ShouldNotBeNil)
	c.So(err.Error(), ShouldContainSubstring, "not found")
}

func TestIntegrationClient(t *testing.T) {
	Convey("with integration client", t, func(c C) {
		env := os.Getenv("ENVIRONMENT")
		if env == "" {
			env = "staging"
		}

		envSettings, err := loadEnv(env)
		c.So(err, ShouldBeNil)

		protoClient := followsrpc.NewFollowsProtobufClient(fmt.Sprintf("http://%s", envSettings.Host), http.DefaultClient)
		c.So(protoClient, ShouldNotBeNil)

		ctx, cancel := context.WithTimeout(context.Background(), time.Second*15)
		defer cancel()

		// With the change introduced in: 25b4ec4cd9ec19b23e26ee85ae65e33eb08d76a0 here each follow that is created is verified by the users service.
		// We can't pass any random ID's any more. These ID's are rejected by the users service.
		// To solve this, we have created two users in staging that do not follow any one or have any followers.
		// Run the following curl command to create test users  in staging:
		// // curl -X POST \
		// //  http://users-service-staging.us-west2.twitch.tv/users \
		// //  -H 'Cache-Control: no-cache' \
		// //  -H 'Content-Type: application/json' \
		// //  -H 'Postman-Token: ebd78c8e-cca7-4ee6-9d1f-c9a60d0af211' \
		// //  -d '{
		// //				"login": "followoingSvcRandomUser2",
		// //				"ip": "1.1.1.1",
		// //				"birthday": {
		// //					"day": 1,
		// //					"month": 1,
		// //					"year": 1891
		// //				},
		// //				"email": "randomuser1@example.com"
		// //			}'
		// // NOTE: change the login key in the request body to use a new loginID
		randomUserID1 := "265466593"
		t.Log("user ID1", randomUserID1)
		randomUserID2 := "265466594"
		t.Log("user ID2", randomUserID2)
		// be safe and reset all follows just in case previous run of the test left things in a bad state.
		_, _ = protoClient.DeleteFollow(ctx, &followsrpc.FollowReq{
			FromUserId:   randomUserID1,
			TargetUserId: randomUserID2,
		})

		// Chain the convey in one call stack so we can also test cache dirty
		c.Convey("Should start empty", func(c C) {
			userIsEmpty(c, ctx, protoClient, randomUserID1)

			c.Convey("When following someone", func(c C) {
				shouldNotFollow(c, ctx, protoClient, randomUserID1, randomUserID2)
				shouldNotFollow(c, ctx, protoClient, randomUserID2, randomUserID1)
				userIsEmpty(c, ctx, protoClient, randomUserID2)

				_, err = protoClient.UpsertFollow(ctx, &followsrpc.UpsertFollowReq{
					FromUserId:   randomUserID1,
					TargetUserId: randomUserID2,
				})
				time.Sleep(time.Second) // delay to propagate change between two graphDB services
				c.So(err, ShouldBeNil)
				shouldNotFollow(c, ctx, protoClient, randomUserID2, randomUserID1)

				followRes, err := protoClient.GetFollow(ctx, &followsrpc.FollowReq{
					FromUserId:   randomUserID1,
					TargetUserId: randomUserID2,
				})
				c.So(err, ShouldBeNil)
				c.So(followRes.FromUserId, ShouldEqual, randomUserID1)
				c.So(followRes.TargetUserId, ShouldEqual, randomUserID2)

				c.Convey("follow direction should match", func(c C) {
					res, err := protoClient.ListFollows(ctx, &followsrpc.FollowsReq{
						UserId: randomUserID1,
					})
					c.So(err, ShouldBeNil)
					c.So(1, ShouldEqual, len(res.Follows))
					c.So(res.Follows[0].FromUserId, ShouldEqual, randomUserID1)
					c.So(res.Follows[0].TargetUserId, ShouldEqual, randomUserID2)

					res2, err := protoClient.ListFollows(ctx, &followsrpc.FollowsReq{
						UserId: randomUserID2,
					})
					c.So(err, ShouldBeNil)
					c.So(0, ShouldEqual, len(res2.Follows))
					c.Convey("follower direction should match", func(c C) {
						shouldNotFollow(c, ctx, protoClient, randomUserID2, randomUserID1)

						res, err := protoClient.ListFollowers(ctx, &followsrpc.FollowsReq{
							UserId: randomUserID1,
						})
						c.So(err, ShouldBeNil)
						c.So(0, ShouldEqual, len(res.Follows))

						res2, err := protoClient.ListFollowers(ctx, &followsrpc.FollowsReq{
							UserId: randomUserID2,
						})
						c.So(err, ShouldBeNil)
						c.So(1, ShouldEqual, len(res2.Follows))
						c.So(res2.Follows[0].FromUserId, ShouldEqual, randomUserID1)
						c.So(res2.Follows[0].TargetUserId, ShouldEqual, randomUserID2)

						res3, err := protoClient.CountFollowers(ctx, &followsrpc.UserIDReq{
							UserId: randomUserID1,
						})
						c.So(err, ShouldBeNil)
						c.So(0, ShouldEqual, res3.Count)

						res4, err := protoClient.CountFollowers(ctx, &followsrpc.UserIDReq{
							UserId: randomUserID2,
						})
						c.So(err, ShouldBeNil)
						c.So(1, ShouldEqual, res4.Count)

						c.Convey("unfollow should empty out", func(c C) {
							_, err := protoClient.DeleteFollow(ctx, &followsrpc.FollowReq{
								FromUserId:   randomUserID1,
								TargetUserId: randomUserID2,
							})
							time.Sleep(time.Second) // delay between two graphDB service
							c.So(err, ShouldBeNil)
							shouldNotFollow(c, ctx, protoClient, randomUserID1, randomUserID2)
							shouldNotFollow(c, ctx, protoClient, randomUserID2, randomUserID1)
							userIsEmpty(c, ctx, protoClient, randomUserID1)
							userIsEmpty(c, ctx, protoClient, randomUserID2)
						})
					})
				})
			})
		})
	})
}
