package follows_test

import (
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"testing"

	"context"

	"code.justin.tv/feeds/following-service/client/follows"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
)

type ClientSuite struct {
	suite.Suite
	client follows.Client
}

var handler = http.NotFound
var ctx = context.Background()

func (suite *ClientSuite) SetupTest() {
	hs := httptest.NewServer(http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
		handler(rw, req)
	}))
	client, err := follows.NewClient(twitchclient.ClientConf{
		Host: hs.URL,
	})
	suite.Nil(err)
	suite.client = client
}

func (suite *ClientSuite) TestFollowWithError() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		rw.WriteHeader(504)
	}
	err := suite.client.Follow(ctx, "2", "3", nil, nil)
	suite.NotNil(err)
}

func (suite *ClientSuite) TestFollowWithBody() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users/3", req.URL.Path)
		suite.Equal("PUT", req.Method)
		body, err := readBody(req.Body)
		suite.Nil(err)
		suite.Equal("{\"block_notifications\":true}", body)
		rw.WriteHeader(204)
	}
	body := &follows.FollowRequestBody{
		BlockNotifications: true,
	}
	err := suite.client.Follow(ctx, "2", "3", body, nil)
	suite.Nil(err)
}

func (suite *ClientSuite) TestFollowNoBody() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users/3", req.URL.Path)
		suite.Equal("PUT", req.Method)
		body, err := readBody(req.Body)
		suite.Nil(err)
		suite.Equal("null", body)
		rw.WriteHeader(204)
	}
	err := suite.client.Follow(ctx, "2", "3", nil, nil)
	suite.Nil(err)
}

func (suite *ClientSuite) TestUnfollow() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users/3", req.URL.Path)
		suite.Equal("DELETE", req.Method)
		rw.WriteHeader(204)
	}
	err := suite.client.Unfollow(ctx, "2", "3", nil)
	suite.Nil(err)
}

func (suite *ClientSuite) TestUnfollowWithError() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		rw.WriteHeader(504)
	}
	err := suite.client.Unfollow(ctx, "2", "3", nil)
	suite.NotNil(err)
}

func (suite *ClientSuite) TestGetFollowError() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users/3", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "bad json")
		suite.Nil(err)
	}
	_, err := suite.client.GetFollow(ctx, "2", "3", nil)
	suite.NotNil(err)
}

func (suite *ClientSuite) TestGetFollow() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users/3", req.URL.Path)
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "{ \"follow\": { \"followed_at\": \"1969-12-31T18:46:40-08:00\",\n    \"from_user_id\": \"1\",\n    \"target_user_id\": \"2\",\n    \"block_notifications\": true,\n    \"is_hidden\": true\n  } }")
		suite.Nil(err)
	}
	follow, err := suite.client.GetFollow(ctx, "2", "3", nil)
	suite.Nil(err)
	assertFollow(suite.T(), follow)
}

func (suite *ClientSuite) TestGetNoFollow() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users/3", req.URL.Path)
		suite.Equal("GET", req.Method)
		rw.Header().Add("Content-Type", "application/json")
		rw.WriteHeader(404)
		_, err := io.WriteString(rw, "{ \"message\": \"A Follow between 22389274 and 2424 was not found\", \"status\": 404 }")
		suite.Nil(err)
	}
	follow, err := suite.client.GetFollow(ctx, "2", "3", nil)
	suite.NotNil(err)
	suite.Equal("404: A Follow between 22389274 and 2424 was not found", err.Error())
	suite.Nil(follow)
}

func (suite *ClientSuite) TestGetSameFollow() {
	follow, err := suite.client.GetFollow(ctx, "1", "1", nil)
	suite.NotNil(err)
	suite.Equal("404: A Follow between 1 and 1 was not found", err.Error())
	suite.Nil(follow)
}

func (suite *ClientSuite) TestListFollows() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users", req.URL.Path)
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "{\n  \"follows\": [\n    {\n      \"followed_at\": \"1969-12-31T18:46:40-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"2\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    },\n    {\n      \"followed_at\": \"1969-12-31T21:33:20-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"3\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    }\n  ],\n  \"_total\": 2,\n  \"_cursor\": \"A Cursor!\"\n}")
		suite.Nil(err)
	}
	follows, err := suite.client.ListFollows(ctx, "2", nil, nil)
	suite.Nil(err)
	assertFollowList(suite.T(), follows)
}

func (suite *ClientSuite) TestListFollowsError() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "bad json")
		suite.Nil(err)
	}
	_, err := suite.client.ListFollows(ctx, "2", nil, nil)
	suite.NotNil(err)
}

func (suite *ClientSuite) TestListFollowsWithParams() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/users?cursor=lulz&direction=asc&limit=40", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "{\n  \"follows\": [\n    {\n      \"followed_at\": \"1969-12-31T18:46:40-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"2\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    },\n    {\n      \"followed_at\": \"1969-12-31T21:33:20-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"3\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    }\n  ],\n  \"_total\": 2,\n  \"_cursor\": \"A Cursor!\"\n}")
		suite.Nil(err)
	}
	params := &follows.ListFollowsParams{
		Limit:         40,
		Cursor:        "lulz",
		SortDirection: "asc",
	}
	follows, err := suite.client.ListFollows(ctx, "2", params, nil)
	suite.Nil(err)
	assertFollowList(suite.T(), follows)
}

func (suite *ClientSuite) TestListFollowers() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/followers/2/users", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "{\n  \"followers\": [\n    {\n      \"followed_at\": \"1969-12-31T18:46:40-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"2\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    },\n    {\n      \"followed_at\": \"1969-12-31T21:33:20-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"3\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    }\n  ],\n  \"_total\": 2,\n  \"_cursor\": \"A Cursor!\"\n}")
		suite.Nil(err)
	}
	followers, err := suite.client.ListFollowers(ctx, "2", nil, nil)
	suite.Nil(err)
	assertFollowerList(suite.T(), followers)
}

func (suite *ClientSuite) TestCountFollowers() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/followers/2/users/count", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "{\n  \"count\": 5\n}")
		suite.Nil(err)
	}
	count, err := suite.client.CountFollowers(ctx, "2", nil)
	suite.Nil(err)
	suite.Equal(count, 5)
}

func (suite *ClientSuite) TestBulkCountFollowers() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Regexp("/v1/followers/[12]/users/count", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		if req.URL.String() == "/v1/followers/1/users/count" {
			_, err := io.WriteString(rw, "{\n  \"count\": 3\n}")
			suite.Nil(err)
		} else {
			_, err := io.WriteString(rw, "{\n  \"count\": 5\n}")
			suite.Nil(err)
		}
	}
	countMap, err := suite.client.BulkCountFollowers(ctx, []string{"1", "2"}, nil)
	suite.Nil(err)
	suite.Equal(2, len(countMap))
	suite.Equal(countMap["2"], 5)
	suite.Equal(countMap["1"], 3)
}

func (suite *ClientSuite) TestListFollowersError() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/followers/2/users", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "bad json")
		suite.Nil(err)
	}
	_, err := suite.client.ListFollowers(ctx, "2", nil, nil)
	suite.NotNil(err)
}

func (suite *ClientSuite) TestRestoreAllFollows() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/restore", req.URL.String())
		suite.Equal("POST", req.Method)
		rw.WriteHeader(204)
	}
	err := suite.client.RestoreAllFollows(ctx, "2", nil)
	suite.Nil(err)
}

func (suite *ClientSuite) TestRestoreAllFollowsError() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2/restore", req.URL.String())
		suite.Equal("POST", req.Method)
		rw.WriteHeader(500)
		_, err := io.WriteString(rw, "bad")
		suite.Nil(err)
	}
	err := suite.client.RestoreAllFollows(ctx, "2", nil)
	suite.NotNil(err)
}

func (suite *ClientSuite) TestHideAllFollows() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2", req.URL.String())
		suite.Equal("DELETE", req.Method)
		rw.WriteHeader(204)
	}
	err := suite.client.HideAllFollows(ctx, "2", nil)
	suite.Nil(err)
}

func (suite *ClientSuite) TestHideAllFollowsError() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/follows/2", req.URL.String())
		suite.Equal("DELETE", req.Method)
		rw.WriteHeader(500)
		_, err := io.WriteString(rw, "bad")
		suite.Nil(err)
	}
	err := suite.client.HideAllFollows(ctx, "2", nil)
	suite.NotNil(err)
}

func (suite *ClientSuite) TestListFollowersWithParams() {
	handler = func(rw http.ResponseWriter, req *http.Request) {
		suite.Equal("/v1/followers/2/users?cursor=lulz&direction=asc&limit=40&offset=10", req.URL.String())
		suite.Equal("GET", req.Method)
		rw.WriteHeader(200)
		_, err := io.WriteString(rw, "{\n  \"followers\": [\n    {\n      \"followed_at\": \"1969-12-31T18:46:40-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"2\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    },\n    {\n      \"followed_at\": \"1969-12-31T21:33:20-08:00\",\n      \"from_user_id\": \"1\",\n      \"target_user_id\": \"3\",\n      \"block_notifications\": true,\n      \"_cursor\": \"A Cursor!\",\n      \"is_hidden\": true\n    }\n  ],\n  \"_total\": 2,\n  \"_cursor\": \"A Cursor!\"\n}")
		suite.Nil(err)
	}
	params := &follows.ListFollowersParams{
		Limit:         40,
		Cursor:        "lulz",
		SortDirection: "asc",
		Offset:        10,
	}
	followers, err := suite.client.ListFollowers(ctx, "2", params, nil)
	suite.Nil(err)
	assertFollowerList(suite.T(), followers)
}

func TestClientSuite(t *testing.T) {
	suite.Run(t, new(ClientSuite))
}

func assertFollowList(t *testing.T, follows *follows.Follows) {
	assert.NotNil(t, follows)
	assert.NotEmpty(t, follows.Cursor)
	assert.NotEmpty(t, follows.Follows)
	for _, follow := range follows.Follows {
		assertFollow(t, follow)
	}
}

func assertFollowerList(t *testing.T, followers *follows.Followers) {
	assert.NotNil(t, followers)
	assert.NotEmpty(t, followers.Cursor)
	assert.NotEmpty(t, followers.Followers)
	for _, follower := range followers.Followers {
		assertFollow(t, follower)
	}
}

func assertFollow(t *testing.T, follow *follows.Follow) {
	assert.NotNil(t, follow)
	assert.False(t, follow.FollowedAt.IsZero())
	assert.NotEmpty(t, follow.FromUserID)
	assert.NotEmpty(t, follow.TargetUserID)
	assert.True(t, follow.BlockNotifications)
}

func readBody(reader io.Reader) (string, error) {
	bytes, err := ioutil.ReadAll(reader)
	if err != nil {
		return "", err
	}
	return string(bytes), nil
}
