package clients

import (
	"context"
	"log"
	"time"

	"code.justin.tv/common/golibs/errorlogger"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/client/publisher"
	"code.justin.tv/eventbus/schema/pkg/user_follow_user"
	"code.justin.tv/feeds/distconf"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/timestamp"
)

var (
	eventBusStatsName = "clients.eventbus"
)

type EventBusClient interface {
	SendFollowerCreatedMessage(ctx context.Context, targetUserID string, fromUserID string, targetFollowerCount int, errorLogger errorlogger.ErrorLogger) error
	SendFollowerDeletedMessage(ctx context.Context, targetUserID string, fromUserID string, targetFollowerCount int, errorLogger errorlogger.ErrorLogger) error
}

// EventBusPublisher exists for unit testing
type EventBusPublisher interface {
	Publish(ctx context.Context, message eventbus.Message) error
}

func NewEventBusClient(environment string, secrets *distconf.Distconf, config *distconf.Distconf, stats statsd.Statter) (EventBusClient, error) {
	region := regionName
	var env publisher.Environment

	conf := aws.NewConfig().WithRegion(region)

	switch environment {
	case "production":
		env = publisher.EnvProduction
	case "staging":
		env = publisher.EnvStaging
	default:
		env = publisher.EnvLocal

		// NewSession uses the instance role creds automatically from
		// https://github.com/aws/aws-sdk-go/blob/master/aws/defaults/defaults.go#L37
		// so we only need to assign creds when in local testing mode
		conf = conf.WithCredentials(credentials.NewStaticCredentials("AKIAFAKE", "fake", ""))
	}

	session, err := session.NewSession(conf)
	if err != nil {
		return nil, err
	}

	base, err := publisher.New(publisher.Config{
		Session:     session,
		Environment: env,
		EventTypes:  []string{user_follow_user.CreateEventType, user_follow_user.DeleteEventType},
	})

	if err != nil {
		return nil, err
	}

	return &eventBusImpl{
		baseClient: base,
		stats:      stats,
	}, nil
}

type eventBusImpl struct {
	baseClient eventbus.Publisher
	stats      statsd.Statter
}

func (p *eventBusImpl) SendFollowerCreatedMessage(ctx context.Context, targetUserID string, fromUserID string, targetFollowerCount int, errorLogger errorlogger.ErrorLogger) error {
	createdAt, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		errorLogger.Error(err)
	}
	return p.sendFollowerCreatedMessage(ctx, targetUserID, fromUserID, targetFollowerCount, createdAt, errorLogger)
}

func (p *eventBusImpl) sendFollowerCreatedMessage(ctx context.Context, targetUserID string, fromUserID string, targetFollowerCount int, createdAt *timestamp.Timestamp, errorLogger errorlogger.ErrorLogger) error {
	err := p.stats.Inc(eventBusStatsName+".send_follower_created_message", 1, defaultStatSampleRate)
	if err != nil {
		errorLogger.Error(err)
		log.Printf("error sending send_follower_created_message counter: %s", err.Error())
	}

	message := &user_follow_user.Create{
		FromUserId:          fromUserID,
		ToUserId:            targetUserID,
		CreatedAt:           createdAt,
		ToUserFollowerCount: int32(targetFollowerCount),
	}

	return hystrix.Do(HystrixEventBusPublish, func() error {
		return p.baseClient.Publish(ctx, message)
	}, nil)
}

func (p *eventBusImpl) SendFollowerDeletedMessage(ctx context.Context, targetUserID string, fromUserID string, targetFollowerCount int, errorLogger errorlogger.ErrorLogger) error {
	deletedAt, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		errorLogger.Error(err)
	}
	return p.sendFollowerDeletedMessage(ctx, targetUserID, fromUserID, targetFollowerCount, deletedAt, errorLogger)
}

func (p *eventBusImpl) sendFollowerDeletedMessage(ctx context.Context, targetUserID string, fromUserID string, targetFollowerCount int, deletedAt *timestamp.Timestamp, errorLogger errorlogger.ErrorLogger) error {
	err := p.stats.Inc(eventBusStatsName+".send_follower_deleted_message", 1, defaultStatSampleRate)
	if err != nil {
		errorLogger.Error(err)
		log.Printf("error sending send_follower_deleted_message counter: %s", err.Error())
	}

	message := &user_follow_user.Delete{
		FromUserId:          fromUserID,
		ToUserId:            targetUserID,
		DeletedAt:           deletedAt,
		ToUserFollowerCount: int32(targetFollowerCount),
	}

	return hystrix.Do(HystrixEventBusPublish, func() error {
		return p.baseClient.Publish(ctx, message)
	}, nil)
}
