package clients

import (
	"errors"
	"os"
	"testing"
	"time"

	"code.justin.tv/common/config"
	"code.justin.tv/eventbus/schema/pkg/user_follow_user"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/following-service/clients/mocks"

	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type EventBusSuite struct {
	suite.Suite
	conf      *distconf.Distconf
	client    *eventBusImpl
	publisher *mocks.EventBusPublisher
}

func (s *EventBusSuite) SetupTest() {
	s.conf = refreshConf()
	err := os.Setenv("aws_access_key_id", "key")
	s.Nil(err)
	err = os.Setenv("aws_secret_access_key", "secret")
	s.Nil(err)

	publisherMock := &mocks.EventBusPublisher{}
	ebClient := &eventBusImpl{
		baseClient: publisherMock,
		stats:      config.Statsd(),
	}

	s.client = ebClient
	s.publisher = publisherMock
}

func (s *EventBusSuite) TestSendFollowerCreatedSuccess() {
	s.publisher.On("Publish", mock.Anything, mock.Anything).Return(nil)
	createdAt, err := ptypes.TimestampProto(time.Now())
	s.Nil(err)
	err = s.client.sendFollowerCreatedMessage(context.Background(), "1", "2", 3, createdAt, NewNoopErrorLogger())
	s.Nil(err)
	expected := &user_follow_user.UserFollowUserCreate{
		FromUserId:          "2",
		ToUserId:            "1",
		CreatedAt:           createdAt,
		ToUserFollowerCount: 3,
	}

	s.publisher.AssertCalled(s.T(), "Publish", mock.Anything, expected)
}

func (s *EventBusSuite) TestSendFollowerCreatedPublishFailure() {
	s.publisher.On("Publish", mock.Anything, mock.Anything).Return(errors.New("publish failure"))
	createdAt, err := ptypes.TimestampProto(time.Now())
	s.Nil(err)
	err = s.client.sendFollowerCreatedMessage(context.Background(), "1", "2", 3, createdAt, NewNoopErrorLogger())
	s.Error(err)
	s.Equal("publish failure", err.Error())
}

func (s *EventBusSuite) TestSendFollowerDeletedSuccess() {
	s.publisher.On("Publish", mock.Anything, mock.Anything).Return(nil)
	deletedAt, err := ptypes.TimestampProto(time.Now())
	s.Nil(err)
	err = s.client.sendFollowerDeletedMessage(context.Background(), "1", "2", 3, deletedAt, NewNoopErrorLogger())
	s.Nil(err)
	expected := &user_follow_user.UserFollowUserDelete{
		FromUserId:          "2",
		ToUserId:            "1",
		DeletedAt:           deletedAt,
		ToUserFollowerCount: 3,
	}

	s.publisher.AssertCalled(s.T(), "Publish", mock.Anything, expected)
}

func (s *EventBusSuite) TestSendFollowerDeletedPublishFailure() {
	s.publisher.On("Publish", mock.Anything, mock.Anything).Return(errors.New("publish failure"))
	deletedAt, err := ptypes.TimestampProto(time.Now())
	s.Nil(err)
	err = s.client.sendFollowerDeletedMessage(context.Background(), "1", "2", 3, deletedAt, NewNoopErrorLogger())
	s.Error(err)
	s.Equal("publish failure", err.Error())
}

func TestEventBusSuite(t *testing.T) {
	suite.Run(t, new(EventBusSuite))
}
