package clients

import (
	"context"
	"fmt"

	followbotDetectionAPI "code.justin.tv/amzn/TwitchFollowBotDetectionTwirp"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/pkg/errors"
)

type FollowBotDetectionClient interface {
	Classify(ctx context.Context, fromUserID string, targetUserID string, clientID string, deviceID string) (int32, error)
}

type followBotDetectionClientImpl struct {
	FollowBotDetection followbotDetectionAPI.TwitchFollowBotDetection
}

func NewFollowBotDetectionClient(fb followbotDetectionAPI.TwitchFollowBotDetection) FollowBotDetectionClient {
	return &followBotDetectionClientImpl{
		FollowBotDetection: fb,
	}
}

func (f *followBotDetectionClientImpl) Classify(ctx context.Context, fromUserID string, targetUserID string, clientID string, deviceID string) (int32, error) {
	classification := int32(followbotDetectionAPI.Tier_UNKNOWN)
	err := hystrix.Do(HystrixFollowBotDetectionClassify, func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%s circuit panic=%v", HystrixFollowBotDetectionClassify, p)
			}
		}()

		resp, err := f.FollowBotDetection.Classify(ctx, &followbotDetectionAPI.Request{
			UserId:       fromUserID,
			TargetUserId: targetUserID,
			ClientId:     clientID,
			DeviceId:     deviceID,
		})
		if err != nil {
			return errors.Wrap(err, "failure calling followbot service")
		}
		if resp != nil {
			classification = int32(resp.Result.Tier)
		}

		return nil
	}, nil)
	if err != nil {
		return classification, err
	}

	return classification, nil
}
