package clients

import (
	"encoding/json"
	"errors"
	"fmt"
	"time"

	pubsub "code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/models"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

const (
	sendFollowStatsName   = "sendfollow.pubsub"
	sendFollowerStatsName = "sendfollower.pubsub"
	sendUnfollowStatsName = "sendunfollow.pubsub"

	// pubsubFollowTargetTopicPrefix is the prefix for the topic that a channel
	// owner can use to see their new follows live
	pubsubFollowTargetTopicPrefix = "following."
	// pubsubFollowerTopicPrefix is the prefix for the topic that a user can
	// use to see their own follows and unfollows.
	pubsubFollowerTopicPrefix = "follows."

	pubsubUserFollowedMessageType   = "user-followed"
	pubsubUserUnfollowedMessageType = "user-unfollowed"
)

// PubSubClient is the interface to Chat's PubSub
type PubSubClient interface {
	SendFollowMessages(ctx context.Context, fromUser *models.Properties, targetUser *models.Properties, errorLogger errorlogger.ErrorLogger) error
	SendFollowMessageToTarget(ctx context.Context, followMessage FollowMessage, targetUserID string, errorLogger errorlogger.ErrorLogger) error
	SendUnfollowMessageToFollower(ctx context.Context, targetUserID string, followerID string, errorLogger errorlogger.ErrorLogger) error
	SendFollowMessageToFollower(ctx context.Context, followerMessage FollowerMessage, followerID string, errorLogger errorlogger.ErrorLogger) error
}

// PubSub implements the Publish function
type PubSub interface {
	Publish(context.Context, []string, string, *twitchclient.ReqOpts) error
}

type pubsubImpl struct {
	baseClient  PubSub
	stats       statsd.Statter
	marshalJSON func(interface{}) ([]byte, error)
}

// NewPubSubClient initializes and returns a new PubSub client
func NewPubSubClient(pubsubHostURL string, stats statsd.Statter) (PubSubClient, error) {
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  pubsubHostURL,
		Stats: stats,
	}

	pubsubClient, err := pubsub.NewPubClient(clientConf)
	if err != nil {
		return nil, errors.New("failed to start pubsub client")
	}

	return &pubsubImpl{
		baseClient:  pubsubClient,
		stats:       stats,
		marshalJSON: json.Marshal,
	}, nil
}

func (p *pubsubImpl) SendFollowMessages(ctx context.Context, fromUser *models.Properties, targetUser *models.Properties, errorLogger errorlogger.ErrorLogger) error {
	if fromUser == nil || targetUser == nil {
		return errors.New("tried to send pubsub messages without both users")
	}
	err := p.SendFollowMessageToTarget(ctx, FollowMessage{
		UserID:      fromUser.ID,
		Username:    fromUser.Login,
		DisplayName: fromUser.Displayname,
	}, targetUser.ID, errorLogger)
	if err != nil {
		return err
	}

	err = p.SendFollowMessageToFollower(ctx, FollowerMessage{
		Type:              pubsubUserFollowedMessageType,
		Timestamp:         time.Now(),
		TargetUserID:      targetUser.ID,
		TargetUsername:    targetUser.Login,
		TargetDisplayName: targetUser.Displayname,
	}, fromUser.ID, errorLogger)
	if err != nil {
		return err
	}

	return nil
}

// FollowMessage is the public-facing pubsub message.
type FollowMessage struct {
	DisplayName *string `json:"display_name"`
	Username    *string `json:"username"`
	UserID      string  `json:"user_id"`
}

// SendFollowPubSubMessageToTarget sends a pubsub message to a topic which
// lists all users who follow the target
func (p *pubsubImpl) SendFollowMessageToTarget(ctx context.Context, followMessage FollowMessage, targetUserID string, errorLogger errorlogger.ErrorLogger) error {
	err := p.stats.Inc(fmt.Sprintf("%s.send_follow_pubsub_message", sendFollowStatsName), 1, defaultStatSampleRate)
	if err != nil {
		errorLogger.Error(err)
	}

	pubsubMessage, err := p.marshalJSON(followMessage)
	if err != nil {
		return err
	}

	pubsubTopic := []string{pubsubFollowTargetTopicPrefix + targetUserID}

	return hystrix.Do(HystrixPubSubPublish, func() error {
		return p.baseClient.Publish(ctx, pubsubTopic, string(pubsubMessage), &twitchclient.ReqOpts{})
	}, nil)
}

// FollowerMessage is the public-facing pubsub message.
type FollowerMessage struct {
	Type              string    `json:"type"`
	Timestamp         time.Time `json:"timestamp"`
	TargetDisplayName *string   `json:"target_display_name"`
	TargetUsername    *string   `json:"target_username"`
	TargetUserID      string    `json:"target_user_id"`
}

func (p *pubsubImpl) SendFollowMessageToFollower(ctx context.Context, followerMessage FollowerMessage, followerID string, errorLogger errorlogger.ErrorLogger) error {
	err := p.stats.Inc(fmt.Sprintf("%s.send_follower_pubsub_message", sendFollowerStatsName), 1, defaultStatSampleRate)
	if err != nil {
		errorLogger.Error(err)
	}

	pubsubMessage, err := p.marshalJSON(followerMessage)
	if err != nil {
		return err
	}

	pubsubTopic := []string{pubsubFollowerTopicPrefix + followerID}

	return hystrix.Do(HystrixPubSubPublish, func() error {
		return p.baseClient.Publish(ctx, pubsubTopic, string(pubsubMessage), &twitchclient.ReqOpts{})
	}, nil)
}

type UnfollowMessage struct {
	Type         string    `json:"type"`
	Timestamp    time.Time `json:"timestamp"`
	TargetUserID string    `json:"target_user_id"`
}

func (p *pubsubImpl) SendUnfollowMessageToFollower(ctx context.Context, targetUserID string, followerID string, errorLogger errorlogger.ErrorLogger) error {
	err := p.stats.Inc(fmt.Sprintf("%s.send_unfollow_pubsub_message", sendUnfollowStatsName), 1, defaultStatSampleRate)
	if err != nil {
		errorLogger.Error(err)
	}

	pubsubMessage, err := p.marshalJSON(UnfollowMessage{
		Type:         pubsubUserUnfollowedMessageType,
		Timestamp:    time.Now(),
		TargetUserID: targetUserID,
	})
	if err != nil {
		return err
	}

	pubsubTopic := []string{pubsubFollowerTopicPrefix + followerID}

	return hystrix.Do(HystrixPubSubPublish, func() error {
		return p.baseClient.Publish(ctx, pubsubTopic, string(pubsubMessage), &twitchclient.ReqOpts{})
	}, nil)
}
