package clients

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/common/config"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/following-service/clients/mocks"
	"code.justin.tv/foundation/twitchclient"

	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type PubSubSuite struct {
	suite.Suite
	conf   *distconf.Distconf
	client *pubsubImpl
	pubsub *mocks.PubSub
}

func (s *PubSubSuite) SetupTest() {
	client, err := NewPubSubClient("pubsubHostURL", config.Statsd())
	s.NoError(err)

	pubsubMock := &mocks.PubSub{}
	pubsubClient, ok := client.(*pubsubImpl)
	s.True(ok)

	pubsubClient.baseClient = pubsubMock
	s.client = pubsubClient
	s.pubsub = pubsubMock
}

func (s *PubSubSuite) TestNew() {
	client, err := NewPubSubClient("pubsubHostURL", nil)
	s.NoError(err)
	s.NotNil(client)
}

func (s *PubSubSuite) IgnoreTestSendFollowToTargetPubSubSuccess() {
	s.pubsub.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil)

	ctx := context.Background()
	login := "test"
	followMessage := FollowMessage{
		Username:    &login,
		DisplayName: &login,
		UserID:      "2",
	}
	err := s.client.SendFollowMessageToTarget(ctx, followMessage, "1", NewNoopErrorLogger())
	s.NoError(err)
	pubsubFollowTopic := []string{"following.1"}
	pubsubMessage := `{"display_name":"test","username":"test","user_id":"2"}`

	s.pubsub.AssertCalled(s.T(), "Publish", ctx, pubsubFollowTopic, pubsubMessage, &twitchclient.ReqOpts{})
}

func (s *PubSubSuite) TestSendFollowToTargetJSONFailure() {
	s.pubsub.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil)
	s.client.marshalJSON = func(v interface{}) ([]byte, error) {
		return nil, errors.New("marshal error")
	}
	err := s.client.SendFollowMessageToTarget(context.Background(), FollowMessage{}, "1", NewNoopErrorLogger())
	s.NotNil(err)
	s.Equal("marshal error", err.Error())
}

func (s *PubSubSuite) TestSendFollowToTargetPublishFailure() {
	s.pubsub.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(errors.New("publish failure"))
	err := s.client.SendFollowMessageToTarget(context.Background(), FollowMessage{}, "1", NewNoopErrorLogger())
	s.NotNil(err)
	s.EqualError(err, "publish failure")
}

func (s *PubSubSuite) TestSendFollowToFollowerPubSubSuccess() {
	s.pubsub.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil)

	ctx := context.Background()
	login := "test"
	followMessage := FollowerMessage{
		Type:              "user-followed",
		Timestamp:         time.Unix(10000, 0),
		TargetUsername:    &login,
		TargetDisplayName: &login,
		TargetUserID:      "2",
	}
	err := s.client.SendFollowMessageToFollower(ctx, followMessage, "1", NewNoopErrorLogger())
	s.NoError(err)
	pubsubFollowTopic := []string{"follows.1"}
	pubsubMessage := `{"type":"user-followed","timestamp":"1969-12-31T18:46:40-08:00","target_display_name":"test","target_username":"test","target_user_id":"2"}`

	s.pubsub.AssertCalled(s.T(), "Publish", ctx, pubsubFollowTopic, pubsubMessage, &twitchclient.ReqOpts{})
}

func (s *PubSubSuite) TestSendFollowToFollowerJSONFailure() {
	s.pubsub.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil)
	s.client.marshalJSON = func(v interface{}) ([]byte, error) {
		return nil, errors.New("marshal error")
	}
	err := s.client.SendFollowMessageToFollower(context.Background(), FollowerMessage{}, "1", NewNoopErrorLogger())
	s.NotNil(err)
	s.Equal("marshal error", err.Error())
}

func (s *PubSubSuite) TestSendFollowToFollowerPublishFailure() {
	s.pubsub.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(errors.New("publish failure"))
	err := s.client.SendFollowMessageToFollower(context.Background(), FollowerMessage{}, "1", NewNoopErrorLogger())
	s.NotNil(err)
	s.EqualError(err, "publish failure")
}
func IgnoreTestPubSubSuite(t *testing.T) {
	suite.Run(t, new(PubSubSuite))
}
