package clients

import (
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"time"

	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/feeds/distconf"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

var (
	followerCreated          = "followerCreatedTopicARN"
	followerCreatedStatsName = "clients.sns"
	newFollowerCreatedEvent  = "followeradd"

	updateFollow          = "updateFollowTopicARN"
	updateFollowStatsName = "updatefollow.sns"
	newUpdateFollowEvent  = "updatefollow"

	regionName = "us-west-2"
)

const (
	defaultStatSampleRate = 1.0
	maxSnsRetries         = 5
)

// SNSClient is the interface to AWS SNS
type SNSClient interface {
	SendUpdateFollowMessage(ctx context.Context, fromUserID string, targetUserID string, event string, targetFollowerCount *int, errorLogger errorlogger.ErrorLogger) error
}

// SNS exists for unit testing
type SNS interface {
	Publish(*sns.PublishInput) (*sns.PublishOutput, error)
}

type snsImpl struct {
	baseClient     SNS
	rolloutPercent *distconf.Float
	stats          statsd.Statter
	topicARNs      map[string]string
	marshalJSON    func(interface{}) ([]byte, error)
	MaxRetries     int
}

type followerCreatedMessage struct {
	FollowerID  string `json:"follower_id"`
	RecipientID string `json:"recipient_id"`
	EmailType   string `json:"email_type"`
}

type updateFollowMessage struct {
	FromUserID          string    `json:"from_id"`
	TargetUserID        string    `json:"target_id"`
	Event               string    `json:"event"`
	UpdatedAt           time.Time `json:"updated_at"`
	TargetFollowerCount *int      `json:"target_follower_count,omitempty"`
}

// NewSNSClient initializes and returns a new SNS client
func NewSNSClient(followerCreatedTopicARN, updateFollowTopicARN string, secrets *distconf.Distconf, config *distconf.Distconf, stats statsd.Statter) (SNSClient, error) {
	if followerCreatedTopicARN == "" {
		return nil, errors.New("Attempting to start following service without new follower topicARN")
	}
	if updateFollowTopicARN == "" {
		return nil, errors.New("Attempting to start following service without update follow topicARN")
	}

	region := regionName
	accessKey := secrets.Str("aws_access_key_id", "").Get()
	if accessKey == "" {
		return nil, errors.New("Attempting to start following service without AWS access key")
	}
	secretKey := secrets.Str("aws_secret_access_key", "").Get()
	if secretKey == "" {
		return nil, errors.New("Attempting to start following service without AWS secret key")
	}
	creds := credentials.NewStaticCredentials(accessKey, secretKey, "")
	svc := sns.New(session.New(), aws.NewConfig().WithRegion(region).WithCredentials(creds))

	arns := map[string]string{
		followerCreated: followerCreatedTopicARN,
		updateFollow:    updateFollowTopicARN,
	}

	return &snsImpl{
		baseClient:  svc,
		stats:       stats,
		topicARNs:   arns,
		marshalJSON: json.Marshal,
		MaxRetries:  maxSnsRetries,
	}, nil
}

func (p *snsImpl) SendUpdateFollowMessage(ctx context.Context, targetUserID string, fromUserID string, event string, targetFollowerCount *int, errorLogger errorlogger.ErrorLogger) error {
	updatedAt := time.Now()
	return p.sendUpdateFollowMessage(ctx, targetUserID, fromUserID, event, updatedAt, targetFollowerCount, errorLogger)
}

func (p *snsImpl) sendUpdateFollowMessage(ctx context.Context, targetUserID string, fromUserID string, event string, updatedAt time.Time, targetFollowerCount *int, errorLogger errorlogger.ErrorLogger) error {
	err := p.stats.Inc(fmt.Sprintf("%s.send_update_follow_message", updateFollowStatsName), 1, defaultStatSampleRate)
	if err != nil {
		errorLogger.Error(err)
		log.Printf("error sending send_update_follow_message counter")
	}

	messageData := &updateFollowMessage{
		FromUserID:          fromUserID,
		TargetUserID:        targetUserID,
		Event:               event,
		UpdatedAt:           updatedAt,
		TargetFollowerCount: targetFollowerCount,
	}
	message, err := p.marshalJSON(messageData)
	if err != nil {
		return err
	}
	snsMessage := string(message)

	topicARN := p.topicARNs[updateFollow]

	params := &sns.PublishInput{
		Message: &snsMessage,
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				DataType:    aws.String("String"),
				StringValue: &newUpdateFollowEvent,
			},
		},
		TopicArn: &topicARN,
	}

	return hystrix.Do(HystrixSNSPublish, func() error {
		_, err = p.baseClient.Publish(params)
		return err
	}, nil)
}
