package clients

import (
	"errors"
	"os"
	"testing"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/config"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/following-service/clients/mocks"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type SNSSuite struct {
	suite.Suite
	conf   *distconf.Distconf
	client *snsImpl
	sns    *mocks.SNS
}

func (s *SNSSuite) SetupTest() {
	s.conf = refreshConf()
	err := os.Setenv("aws_access_key_id", "key")
	s.Nil(err)
	err = os.Setenv("aws_secret_access_key", "secret")
	s.Nil(err)

	client, err := NewSNSClient("followerCreatedTopic", "updateFollowTopic", s.conf, s.conf, config.Statsd())
	s.Nil(err)
	awsSNSMock := &mocks.SNS{}
	snsClient, ok := client.(*snsImpl)
	s.True(ok)
	snsClient.baseClient = awsSNSMock
	s.client = snsClient
	s.sns = awsSNSMock
}

func (s *SNSSuite) TestNew() {
	client, err := NewSNSClient("followerCreatedTopic", "updateFollowTopic", s.conf, s.conf, nil)
	s.Nil(err)
	s.NotNil(client)
}

func (s *SNSSuite) TestNewRequireFollowerCreatedTopic() {
	client, err := NewSNSClient("", "updateFollowTopic", s.conf, s.conf, nil)
	s.Nil(client)
	s.NotNil(err)
	s.Equal("Attempting to start following service without new follower topicARN", err.Error())
}

func (s *SNSSuite) TestNewRequireUpdateFollowTopic() {
	client, err := NewSNSClient("followerCreatedTopic", "", s.conf, s.conf, nil)
	s.Nil(client)
	s.NotNil(err)
	s.Equal("Attempting to start following service without update follow topicARN", err.Error())
}

func (s *SNSSuite) TestNewRequiresAccessKey() {
	err := os.Setenv("aws_access_key_id", "")
	s.Nil(err)
	client, err := NewSNSClient("followerCreatedTopic", "updateFollowTopic", refreshConf(), s.conf, nil)
	s.Nil(client)
	s.NotNil(err)
	s.Equal("Attempting to start following service without AWS access key", err.Error())
}

func (s *SNSSuite) TestNewRequiresSecretKey() {
	err := os.Setenv("aws_secret_access_key", "")
	s.Nil(err)
	client, err := NewSNSClient("followerCreatedTopic", "updateFollowTopic", refreshConf(), s.conf, nil)
	s.Nil(client)
	s.NotNil(err)
	s.Equal("Attempting to start following service without AWS secret key", err.Error())
}

func (s *SNSSuite) TestSendUpdateFollowSuccess() {
	updatedAt := time.Now()
	updatedAtJSON, err := updatedAt.MarshalJSON()
	s.Nil(err)
	updatedAtJSONString := string(updatedAtJSON)
	s.sns.On("Publish", mock.Anything).Return(&sns.PublishOutput{}, nil)
	zero := 0
	err = s.client.sendUpdateFollowMessage(context.Background(), "1", "2", "3", updatedAt, &zero, NewNoopErrorLogger())
	s.Nil(err)
	expected := &sns.PublishInput{
		Message: aws.String("{\"from_id\":\"2\",\"target_id\":\"1\",\"event\":\"3\",\"updated_at\":" + updatedAtJSONString + ",\"target_follower_count\":0}"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				DataType:    aws.String("String"),
				StringValue: aws.String(newUpdateFollowEvent),
			},
		},
		TopicArn: aws.String("updateFollowTopic"),
	}
	s.sns.AssertCalled(s.T(), "Publish", expected)
}

func (s *SNSSuite) TestSendUpdateFollowJSONFailure() {
	s.sns.On("Publish", mock.Anything).Return(&sns.PublishOutput{}, nil)
	s.client.marshalJSON = func(v interface{}) ([]byte, error) {
		return nil, errors.New("marshal error")
	}
	zero := 0
	err := s.client.sendUpdateFollowMessage(context.Background(), "1", "2", "3", time.Now(), &zero, NewNoopErrorLogger())
	s.NotNil(err)
	s.Equal("marshal error", err.Error())
}

func (s *SNSSuite) TestSendUpdateFollowPublishFailure() {
	s.sns.On("Publish", mock.Anything).Return(&sns.PublishOutput{}, errors.New("publish failure"))
	zero := 0
	err := s.client.sendUpdateFollowMessage(context.Background(), "1", "2", "3", time.Now(), &zero, NewNoopErrorLogger())
	s.NotNil(err)
	s.Equal("publish failure", err.Error())
}

func TestSNSSuite(t *testing.T) {
	suite.Run(t, new(SNSSuite))
}

func refreshConf() *distconf.Distconf {
	return &distconf.Distconf{
		Readers: []distconf.Reader{
			&distconf.Env{},
		},
	}
}
