package clients

import (
	"log"

	"code.justin.tv/hygienic/spade"
	"golang.org/x/net/context"
)

type SpadeClientWrapper interface {
	QueueEvents(...spade.Event)
	Start() error
	Close() error
}

type spadeImpl struct {
	baseClient   SpadeClientWrapper
	closesOnStop chan struct{}
}

const (
	followEventName                   = "server_follow"
	unfollowEventName                 = "server_unfollow"
	notificationStatusUpdateEventName = "server_notification_status_update"
	rolloutPercent                    = 0.2
)

// SpadeClient is the interface for sending spade events
type SpadeClient interface {
	Start()
	TrackFollowEvent(ctx context.Context, event *SpadeEventData)
	TrackUnfollowEvent(ctx context.Context, event *SpadeEventData)
	TrackNotificationStatusUpdateEvent(ctx context.Context, event *SpadeEventData)
	FollowCountRolloutPercent() float64
}

// SpadeEventData are the fields required to send a
// `server_follow` or `server_unfollow`` event
type SpadeEventData struct {
	ClientID            string `json:"client_id"`
	DeviceID            string `json:"device_id"`
	FromUserID          string `json:"from_user_id"`
	TargetUserID        string `json:"target_user_id"`
	FollowCount         *int   `json:"follow_count,omitempty"`
	AllowsNotifications *bool  `json:"allows_notifications,omitempty"`
}

// NewSpadeClient creates a new spade client
func NewSpadeClient(closesOnStop chan struct{}) (SpadeClient, error) {
	spadeClient := spade.Client{
		Config: &spade.StaticConfig{},
	}

	err := spadeClient.Setup()
	if err != nil {
		return nil, err
	}

	return &spadeImpl{
		baseClient:   &spadeClient,
		closesOnStop: closesOnStop,
	}, nil
}

func (s *spadeImpl) Start() {
	go s.listenForStop()
	err := s.baseClient.Start()
	if err != nil {
		log.Printf("error starting spade client: %v", err)
	}
}

func (s *spadeImpl) listenForStop() {
	<-s.closesOnStop
	err := s.baseClient.Close()
	if err != nil {
		log.Printf("error closing spade client: %v", err)
	}
}

// FollowCountRolloutPercent is the percentage rollout for querying
// `follow_count` on the `server_follow` and `server_unfollow` events
func (s *spadeImpl) FollowCountRolloutPercent() float64 {
	return rolloutPercent
}

func (s *spadeImpl) TrackFollowEvent(ctx context.Context, event *SpadeEventData) {
	s.trackSpadeEvent(ctx, followEventName, event)
}

func (s *spadeImpl) TrackUnfollowEvent(ctx context.Context, event *SpadeEventData) {
	s.trackSpadeEvent(ctx, unfollowEventName, event)
}

func (s *spadeImpl) TrackNotificationStatusUpdateEvent(ctx context.Context, event *SpadeEventData) {
	s.trackSpadeEvent(ctx, notificationStatusUpdateEventName, event)
}

func (s *spadeImpl) trackSpadeEvent(ctx context.Context, eventName string, eventData *SpadeEventData) {
	s.baseClient.QueueEvents(spade.Event{
		Name:       eventName,
		Properties: eventData,
	})
}
