package clients

import (
	"testing"

	"code.justin.tv/feeds/following-service/clients/mocks"

	"golang.org/x/net/context"

	"code.justin.tv/hygienic/spade"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type SpadeSuite struct {
	suite.Suite
	client *spadeImpl
	spade  *mocks.SpadeClientWrapper
}

func (s *SpadeSuite) SetupTest() {
	fakeSpadeClient := &mocks.SpadeClientWrapper{}
	s.client = &spadeImpl{
		baseClient: fakeSpadeClient,
	}
	s.spade = fakeSpadeClient
	s.spade.On("QueueEvents", mock.Anything).Return()
}

func (s *SpadeSuite) TestNew() {
	closesOnStop := make(chan struct{})
	client, err := NewSpadeClient(closesOnStop)
	s.Nil(err)
	s.NotNil(client)
}

func (s *SpadeSuite) TestTrackFollowEvent() {
	eventData := &SpadeEventData{
		ClientID:     "client1234",
		DeviceID:     "device123",
		FromUserID:   "fromUserID",
		TargetUserID: "targetUserID",
	}

	spadeEvent := []spade.Event{spade.Event{
		Name:       "server_follow",
		Properties: eventData,
	}}

	s.client.TrackFollowEvent(context.Background(), eventData)
	s.spade.AssertCalled(
		s.T(),
		"QueueEvents",
		spadeEvent,
	)
}

func (s *SpadeSuite) TestTrackFollowEventWithFollowerCount() {
	numFollows := 30
	eventData := &SpadeEventData{
		ClientID:     "client1234",
		DeviceID:     "device123",
		FromUserID:   "fromUserID",
		TargetUserID: "targetUserID",
		FollowCount:  &numFollows,
	}

	spadeEvent := []spade.Event{spade.Event{
		Name:       "server_follow",
		Properties: eventData,
	}}

	s.client.TrackFollowEvent(context.Background(), eventData)
	s.spade.AssertCalled(
		s.T(),
		"QueueEvents",
		spadeEvent,
	)
}

func (s *SpadeSuite) TestTrackUnfollowEvent() {
	eventData := &SpadeEventData{
		ClientID:     "client1234",
		DeviceID:     "device123",
		FromUserID:   "fromUserID",
		TargetUserID: "targetUserID",
	}

	spadeEvent := []spade.Event{spade.Event{
		Name:       "server_unfollow",
		Properties: eventData,
	}}

	s.client.TrackUnfollowEvent(context.Background(), eventData)
	s.spade.AssertCalled(
		s.T(),
		"QueueEvents",
		spadeEvent,
	)
}

func (s *SpadeSuite) TestTrackUnfollowEventWithFollowerCount() {
	numFollows := 30
	eventData := &SpadeEventData{
		FromUserID:   "fromUserID",
		TargetUserID: "targetUserID",
		FollowCount:  &numFollows,
	}

	spadeEvent := []spade.Event{spade.Event{
		Name:       "server_unfollow",
		Properties: eventData,
	}}

	s.client.TrackUnfollowEvent(context.Background(), eventData)
	s.spade.AssertCalled(
		s.T(),
		"QueueEvents",
		spadeEvent,
	)
}

func (s *SpadeSuite) TestTrackNotificationStatusUpdateEvent() {
	allowsNotifications := true
	eventData := &SpadeEventData{
		ClientID:            "client1234",
		DeviceID:            "device123",
		FromUserID:          "fromUserID",
		TargetUserID:        "targetUserID",
		AllowsNotifications: &allowsNotifications,
	}

	spadeEvent := []spade.Event{spade.Event{
		Name:       "server_notification_status_update",
		Properties: eventData,
	}}

	s.client.TrackNotificationStatusUpdateEvent(context.Background(), eventData)
	s.spade.AssertCalled(
		s.T(),
		"QueueEvents",
		spadeEvent,
	)
}

func TestSpadeSuite(t *testing.T) {
	suite.Run(t, new(SpadeSuite))
}
