package errors

import (
	"fmt"
	"net/http"
)

// HasHTTPStatus describes an error which specifies an HTTP status
type HasHTTPStatus interface {
	Status() int
}

// RequestBodyError is returned when the body of a request cannot be read or contains
// the wrong type of field.
type RequestBodyError struct {
	ErrorString string
}

// Error returns the string representation of a RequestBodyError.
func (e RequestBodyError) Error() string {
	return e.ErrorString
}

// Status returns 400
func (e RequestBodyError) Status() int {
	return http.StatusBadRequest
}

// InvalidValueError is returned when a parameter is passed in with a value that is not
// part of a valid list.
type InvalidValueError struct {
	ParamName string
	Value     interface{}
}

// Error returns the string representation of a InvalidValueError.
func (e InvalidValueError) Error() string {
	return fmt.Sprintf("Invalid value for %s: %v", e.ParamName, e.Value)
}

// Status returns 400
func (e InvalidValueError) Status() int {
	return http.StatusBadRequest
}

// MissingParamError is returned when a required parameter is missing.
type MissingParamError struct {
	ParamName string
}

// Error returns the string representation of a MissingParamError.
func (e MissingParamError) Error() string {
	return "Missing required param: " + e.ParamName
}

// Status returns 400
func (e MissingParamError) Status() int {
	return http.StatusBadRequest
}

// AssociationNotFoundError is returned when a corresponding association does not exist.
type AssociationNotFoundError struct {
	Type   string
	FromID string
	ToID   string
}

// Error returns the string representation of a NotFoundError.
func (e AssociationNotFoundError) Error() string {
	return fmt.Sprintf("A %s between %s and %s was not found", e.Type, e.FromID, e.ToID)
}

// Status returns 404
func (e AssociationNotFoundError) Status() int {
	return http.StatusNotFound
}

// StatusCodeFrom takes in an error, and matches it to the appropriate status response code.
func StatusCodeFrom(e interface{}) int {
	err, ok := e.(HasHTTPStatus)
	if ok {
		return err.Status()
	}
	return http.StatusInternalServerError
}

// ForbiddenError is returned when the user is not allowed to perform an action
type ForbiddenError struct {
	ErrorString string
}

// Error returns the string representation of a ForbiddenError.
func (e ForbiddenError) Error() string {
	return e.ErrorString
}

// Status returns 403
func (e ForbiddenError) Status() int {
	return http.StatusForbidden
}
