package errors

import (
	"errors"
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestMissingParamError(t *testing.T) {
	error := MissingParamError{ParamName: "Name"}
	assert.Equal(t, http.StatusBadRequest, StatusCodeFrom(error))
	assert.Equal(t, "Missing required param: Name", error.Error())
	assert.Equal(t, http.StatusBadRequest, error.Status())
}

func TestAssociationNotFound(t *testing.T) {
	error := AssociationNotFoundError{Type: "User", FromID: "1", ToID: "2"}
	assert.Equal(t, http.StatusNotFound, StatusCodeFrom(error))
	assert.Equal(t, "A User between 1 and 2 was not found", error.Error())
	assert.Equal(t, http.StatusNotFound, error.Status())
}

func TestInvalidValueError(t *testing.T) {
	error := InvalidValueError{ParamName: "Name", Value: "lulz"}
	assert.Equal(t, http.StatusBadRequest, StatusCodeFrom(error))
	assert.Equal(t, "Invalid value for Name: lulz", error.Error())
	assert.Equal(t, http.StatusBadRequest, error.Status())
}

func TestRequestBodyError(t *testing.T) {
	error := RequestBodyError{ErrorString: "qwer"}
	assert.Equal(t, http.StatusBadRequest, StatusCodeFrom(error))
	assert.Equal(t, "qwer", error.Error())
	assert.Equal(t, http.StatusBadRequest, error.Status())
}

func TestServerError(t *testing.T) {
	error := errors.New("Internal Server Error")
	assert.Equal(t, http.StatusInternalServerError, StatusCodeFrom(error))
}

func TestForbiddenError(t *testing.T) {
	error := ForbiddenError{ErrorString: "not allowed"}
	assert.Equal(t, http.StatusForbidden, StatusCodeFrom(error))
	assert.Equal(t, "not allowed", error.Error())
	assert.Equal(t, http.StatusForbidden, error.Status())
}
