package header

import (
	"context"
	"net/http"
)

const (
	ClientIDHeader = "Twitch-Client-Id"
	DeviceIDHeader = "Device-ID"
)

// add headers (currently client-id and device-id) to context to consume in twirp methods
// ref: https://github.com/twitchtv/twirp/blob/master/docs/headers.md#read-http-headers-from-requests
func WithHeaders(base http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		ctx := r.Context()
		cid := r.Header.Get(ClientIDHeader)
		did := r.Header.Get(DeviceIDHeader)

		ctx = context.WithValue(ctx, ClientIDHeader, cid)
		ctx = context.WithValue(ctx, DeviceIDHeader, did)

		r = r.WithContext(ctx)

		base.ServeHTTP(w, r)
	})
}

// GetHeaderFromContext returns the value of the header provided that was injected
// in the context earlier. Missing headers will return as empty strings ("")
func GetHeaderFromContext(ctx context.Context, header string) string {
	val, ok := ctx.Value(header).(string)
	if ok {
		return val
	}

	return ""
}
