package main

import (
	"context"
	"crypto/rand"
	"fmt"
	"log"
	"math"
	"math/big"
	"net/http"
	"os"
	"time"

	followbotDetectionAPI "code.justin.tv/amzn/TwitchFollowBotDetectionTwirp"
	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	zuma "code.justin.tv/chat/zuma/client"
	"code.justin.tv/common/config"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/common/golibs/errorlogger/rollbar"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/discovery/experiments"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/backend/backendcache"
	"code.justin.tv/feeds/following-service/cache"
	"code.justin.tv/feeds/following-service/clients"
	"code.justin.tv/feeds/following-service/header"
	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"code.justin.tv/feeds/following-service/rpc/followsrpcserver"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/metrics/sfx/sfxhystrix"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdim"
	"code.justin.tv/feeds/sandyconf"
	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/growth/receiver/proto/receiver"
	"code.justin.tv/hygienic/httpheaders"
	usersservice "code.justin.tv/web/users-service/client"
	"github.com/afex/hystrix-go/plugins"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"
	"github.com/twitchtv/twirp"
	twirp_statsd_hook "github.com/twitchtv/twirp/hooks/statsd"
	goji "goji.io"
	"goji.io/pat"
)

const (
	fatalError = 1
)

type configuration struct {
	Secrets        *distconf.Distconf
	Config         *distconf.Distconf
	SignalFxResult *sfxstatsd.SetupResult
}

type service struct {
	server        *Server
	backend       backend.Backender
	logger        errorlogger.ErrorLogger
	dconf         *distconf.Distconf
	osExit        func(code int)
	log           func(line ...interface{})
	runServer     func(goji.Handler, *twitchhttp.ServerConfig) error
	createServer  func(stats statsd.Statter, logger errorlogger.ErrorLogger, backend backend.Backender, exps experiments.Experiments) (*Server, error)
	parseConfig   func() error
	resolveConfig func(parameter string) string
}

// Server is the basic struct which contains the pieces needed to server requests.
type Server struct {
	*goji.Mux
	stats       statsd.Statter
	errorLogger errorlogger.ErrorLogger
}

func (s *service) main() {
	closesOnStop := make(chan struct{})
	defer close(closesOnStop)
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		panic("Unable to get environment. If you're running locally, please use make dev.")
	}

	conf, err := loadConfig(env)
	if err != nil {
		s.log(err)
		s.osExit(fatalError)
		return
	}
	s.dconf = conf.Config

	err = s.configureServer(conf, env, closesOnStop)
	if err != nil {
		s.log(err)
		s.osExit(fatalError)
		return
	}

	fmt.Println(conf.Config.Var().String())

	twitchhttp.AddDefaultSignalHandlers()
	s.log(s.runServer(s.server, nil))
}

const maxDirtyConcurrency = 15000

var instance = &service{
	osExit:       os.Exit,
	log:          log.Fatal,
	runServer:    twitchhttp.ListenAndServe,
	createServer: NewServer,
	parseConfig:  config.Parse,
}

func loadConfig(env string) (*configuration, error) {
	// TODO: remove the remainder of common/config so this is not necessary
	err := config.Parse()
	if err != nil {
		return nil, err
	}
	return loadDistConf(env)
}

func setupSignalFx(conf *configuration) error {
	c := sfxstatsd.SetupConfig{
		AuthToken: conf.Secrets.Str("signalfx.access_token", "").Get(),
		CounterMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "endpoints.follows.*",
				DimensionsMap: "-.-.handler",
				MetricName:    "http.hits",
			},
			{
				MetricPath:    "backendcache.*.*.*.*",
				DimensionsMap: "%.cache_call.dir.size.%",
			},
			{
				MetricPath:    "backendcache.*.*",
				DimensionsMap: "%.cache_call.%",
			},
			{
				MetricPath:    "*.sns.*",
				DimensionsMap: "update_type.method.method",
				MetricName:    "updates_sent",
			},
			{
				MetricPath:    "offset.used",
				DimensionsMap: "%.%",
			},
			{
				MetricPath:    "sendfollow.pubsub.*",
				DimensionsMap: "%.%.method",
			},
		},
		TimingMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "endpoints.follows.*",
				DimensionsMap: "-.-.handler",
				MetricName:    "http.time",
			},
		},
		DefaultDimensions: map[string]string{
			"service": "following-service",
			"env":     os.Getenv("ENVIRONMENT"),
		},
	}
	if c.AuthToken == "" {
		fmt.Println("Not using SignalFx: no auth token")
		return nil
	}
	sfxSetupResult, err := sfxstatsd.SingletonSetupAndInstall(c)
	if err != nil {
		return err
	}
	config.SetStatsd(&sfxSetupResult.Statter)
	conf.SignalFxResult = sfxSetupResult
	sfxhystrix.Setup(sfxSetupResult)
	return nil
}

func (s *service) configureServer(conf *configuration, env string, closesOnStop chan struct{}) error {
	if err := setupSignalFx(conf); err != nil {
		return err
	}

	expsClient, err := clients.NewExperimentClient(s.logger)
	if err != nil {
		return err
	}

	s.logger = configureRollbar(conf, env)

	s.backend, err = configureBackend(conf, env, s.logger, closesOnStop, expsClient)
	if err != nil {
		return errors.Wrap(err, "failed to create backend")
	}

	s.server, err = NewServer(
		config.Statsd(),
		s.logger,
		s.backend,
		expsClient,
	)
	return err
}

// NewServer generates a new server to handle API requests.
func NewServer(stats statsd.Statter, logger errorlogger.ErrorLogger, backend backend.Backender, expsClient experiments.Experiments) (*Server, error) {
	//TODO: deprecate use of goji when moving to Fulton
	gojiMux := twitchhttp.NewServer()
	s := &Server{
		gojiMux,
		stats,
		logger,
	}

	// hooks for stats + error handling
	var statsHook *twirp.ServerHooks
	if stats != nil {
		statsHook = twirp_statsd_hook.NewStatsdServerHooks(stats)
	}
	logInternalErrorsHook := followsrpcserver.NewInternalErrorsLoggerHook(logger)
	hooks := twirp.ChainHooks(statsHook, logInternalErrorsHook)

	// RPC service
	rpcServer := &followsrpcserver.Server{Backend: backend, Stats: stats, Exps: expsClient}
	rpcHandler := followsrpc.NewFollowsServer(rpcServer, hooks)
	s.Handle(pat.Post(followsrpc.FollowsPathPrefix+"*"), header.WithHeaders(rpcHandler))

	return s, nil
}

func setupCache(conf *configuration) (*memcache.Client, error) {
	cfgServer := conf.Config.Str("following-service.cache.config_server", "").Get()
	if cfgServer == "" {
		return nil, nil
	}
	memCache, _ := memcache.Elasticache(cfgServer, time.Minute)
	memCache.MaxIdleConns(5)
	memCache.Prewarm(5)

	cacheKey, cErr := getCacheKey()
	if cErr != nil {
		return nil, errors.Wrap(cErr, "unable to generate cache key")
	}

	if err := memCache.Set(context.Background(), &memcache.Item{
		Key:        cacheKey,
		Value:      []byte("test"),
		Expiration: 60,
	}); err != nil {
		return nil, errors.Wrap(err, "unable to verify memcache client set")
	}
	var item *memcache.Item
	var err error
	if item, err = memCache.Get(context.Background(), cacheKey); err != nil {
		return nil, errors.Wrap(err, "unable to verify memcache client get")
	}

	if string(item.Value) != "test" {
		return nil, errors.New("unable to verify memcache client set-get")
	}
	return memCache, nil
}

func configureBackend(conf *configuration, env string, errorLogger errorlogger.ErrorLogger, closesOnStop chan struct{}, exps experiments.Experiments) (backend.Backender, error) {
	graphdbClient, err := configureGraphDBAPI(conf)
	if err != nil {
		return nil, err
	}

	graphdbFultonClient, err := configureGraphDBFulton(conf)
	if err != nil {
		return nil, err
	}

	err = configureHystrix(conf, env)
	if err != nil {
		return nil, err
	}

	sns, err := configureSNS(conf)
	if err != nil {
		return nil, err
	}

	dart, err := configureDart(conf)
	if err != nil {
		return nil, err
	}

	pubsub, err := configurePubSub(conf)
	if err != nil {
		return nil, err
	}

	usersserviceInstance, err := configureUsersService(conf)
	if err != nil {
		return nil, err
	}

	spade, err := clients.NewSpadeClient(closesOnStop)
	if err != nil {
		return nil, err
	}

	eventbus, err := configureEventBus(conf, env)
	if err != nil {
		return nil, err
	}

	zumaClient, err := createZumaClient(conf)
	if err != nil {
		return nil, err
	}

	followBotDetectionClient, err := createFollowBotDetectionClient(conf)
	if err != nil {
		return nil, err
	}

	go spade.Start()

	dataBackend := &backend.Backend{
		Client:                   graphdbClient,
		GraphDBFultonClient:      graphdbFultonClient,
		ErrorLogger:              errorLogger,
		Stats:                    config.Statsd(),
		SNS:                      sns,
		PubSub:                   pubsub,
		UsersClient:              usersserviceInstance,
		Spade:                    spade,
		EventBus:                 eventbus,
		ZumaClient:               zumaClient,
		DartClient:               dart,
		FollowBotDetectionClient: followBotDetectionClient,
	}

	cacheConfig := cache.Config{}
	err = cacheConfig.Load(conf.Config)
	if err != nil {
		return nil, err
	}
	memCache, err := setupCache(conf)
	if err != nil {
		errorLogger.Error(errors.Wrap(err, "Unable to setup Memcache server.  Falling back to non cached impl"))
		errorLogger.Info("Not using cache client at all")
		return dataBackend, nil
	}
	if memCache == nil {
		errorLogger.Info("Not using cache client.  Configured to empty string")
		return dataBackend, nil
	}
	mcCache := cache.MemcacheCache{
		Config:         &cacheConfig,
		MemcacheClient: memCache,
	}

	cacheLimitConfig := &backendcache.CacheLimitationsConfig{}
	if err := cacheLimitConfig.Load(conf.Config); err != nil {
		return nil, err
	}
	backendCacheClient := &backendcache.BackendCache{
		Cache:   &mcCache,
		Statter: config.Statsd(),
		CacheLimits: backendcache.CacheLimitations{
			Config: cacheLimitConfig,
		},
		DataBackend: dataBackend,
		ErrorLogger: errorLogger,
		DirtyCh:     make(chan string, maxDirtyConcurrency),
		Exps:        exps,
	}
	backendCacheClient.CacheLimits.Reload()
	backendCacheClient.CacheLimits.CacheLimitSizes()
	go backendCacheClient.DirtyLoop()
	go backendCacheClient.Monitor()
	return backendCacheClient, nil
}

func configureHystrix(conf *configuration, env string) error {
	statsdAddr := conf.Config.Str("following-service.statsd_addr", "statsd.central.twitch.a2z.com:8125")
	hostName, err := os.Hostname()
	if err != nil {
		hostName = "unknown"
	}
	return clients.InitHystrix(
		&plugins.StatsdCollectorConfig{
			Prefix:     fmt.Sprintf("following-service.%s.%s.hystrix", env, hostName),
			StatsdAddr: statsdAddr.Get(),
		},
	)
}

func configureRollbar(conf *configuration, env string) errorlogger.ErrorLogger {
	rollbarToken := conf.Secrets.Str("rollbar_token", "")
	return rollbar.NewRollbarLogger(rollbarToken.Get(), env)
}

const HeaderXCallerService = "X-Caller-Service"

func newHTTPClient() *httpheaders.HeaderClient {
	client := &http.Client{
		Transport: http.RoundTripper(&http.Transport{
			MaxIdleConnsPerHost: 100,
		}),
	}
	headerClient := httpheaders.WithHeader(client, HeaderXCallerService, "following-service")
	return headerClient
}

func configureDart(conf *configuration) (clients.DartClient, error) {
	hostname := conf.Config.Str("dart_receiver.hostname", "").Get()
	if hostname == "" {
		return nil, errors.New("unable to find dart hostname")
	}
	client := newHTTPClient()
	return clients.NewDartClient(receiver.NewReceiverProtobufClient(hostname, client)), nil
}

func configureGraphDBAPI(conf *configuration) (graphdb.GraphDB, error) {
	graphdbAddr := conf.Config.Str("graphdb.alb_dns", "").Get()
	if graphdbAddr == "" {
		return nil, errors.New("unable to find graphdb address")
	}
	client := newHTTPClient()
	return graphdb.NewGraphDBProtobufClient(graphdbAddr, client), nil
}

func configureGraphDBFulton(conf *configuration) (graphdbFulton.TwitchVXGraphDBECS, error) {
	graphdbFultonAddr := conf.Config.Str("following-service.graphdb.hostname", "").Get()
	if graphdbFultonAddr == "" {
		return nil, errors.New("unable to find graphdb fulton address")
	}
	client := newHTTPClient()
	return graphdbFulton.NewTwitchVXGraphDBECSProtobufClient(graphdbFultonAddr, client), nil
}

func configureSNS(conf *configuration) (clients.SNSClient, error) {
	followerCreatedTopicARN := conf.Config.Str("following-service.followercreated_topic_arn", "")
	updateFollowTopicARN := conf.Config.Str("following-service.updatefollow_topic_arn", "")
	return clients.NewSNSClient(followerCreatedTopicARN.Get(), updateFollowTopicARN.Get(), conf.Secrets, conf.Config, config.Statsd())
}

func configureEventBus(conf *configuration, env string) (clients.EventBusClient, error) {
	return clients.NewEventBusClient(env, conf.Secrets, conf.Config, config.Statsd())
}

func configurePubSub(conf *configuration) (clients.PubSubClient, error) {
	pubsubHostURL := conf.Config.Str("following-service.pubsub_host_url", "")
	return clients.NewPubSubClient(pubsubHostURL.Get(), config.Statsd())
}

func configureUsersService(conf *configuration) (usersservice.Client, error) {
	usersserviceHostURL := conf.Config.Str("following-service.usersservice_host_url", "")
	return usersservice.NewClient(twitchclient.ClientConf{Host: usersserviceHostURL.Get()})
}

func createZumaClient(conf *configuration) (zuma.Client, error) {
	return zuma.NewClient(twitchclient.ClientConf{
		Host: conf.Config.Str("following-service.zuma_host_url", "").Get(),
	})
}

func createFollowBotDetectionClient(conf *configuration) (clients.FollowBotDetectionClient, error) {
	hostname := conf.Config.Str("following-service.follow_bot_detection.hostname", "").Get()
	if hostname == "" {
		return nil, errors.New("unable to find follow bot detection hostname")
	}

	return clients.NewFollowBotDetectionClient(followbotDetectionAPI.NewTwitchFollowBotDetectionProtobufClient(hostname, newHTTPClient())), nil
}

func loadDistConf(env string) (*configuration, error) {
	jconf := distconf.JSONConfig{}
	err := jconf.RefreshFile(fmt.Sprintf("config/%s.json", env))
	if err != nil {
		return nil, err
	}

	conf := &distconf.Distconf{
		Readers: []distconf.Reader{
			&jconf,
		},
	}

	roleARN := conf.Str("sandstorm.arn", "")
	mc := sandyconf.ManagerConstructor{}
	defaultRegion := "us-west-2" // Ideally this should not be heardcoded.
	mc.AwsConfig = &aws.Config{
		Region:              &defaultRegion,
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	}

	mgr := mc.Manager(roleARN.Get())
	sandyConf := sandyconf.Sandyconf{
		Team:        "feeds",
		Service:     "following-service",
		Environment: env,
		Manager:     mgr,
	}
	secrets := &distconf.Distconf{
		Readers: []distconf.Reader{
			&sandyConf,
		},
	}

	return &configuration{
		Secrets: secrets,
		Config:  conf,
	}, nil
}

func getCacheKey() (string, error) {
	key, err := rand.Int(rand.Reader, big.NewInt(math.MaxInt32))
	if err != nil {
		return "", err
	}

	return fmt.Sprintf("cachetest:%d", key), nil
}

func main() {
	instance.main()
}
