package followsrpc_test

import (
	"testing"

	"context"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/twitchtv/twirp"
)

var ctx = context.Background()

// BulkCountFollowers tests

func Test_BulkCountFollowers_WithEmptyUserIds_ReturnsEmptyList(t *testing.T) {
	mockClient := &fakeFollowsCountClient{}

	userIDs := []string{} // empty list
	respMap, err := followsrpc.BulkCountFollowers(ctx, mockClient, userIDs)
	assert.NoError(t, err)
	assert.Equal(t, 0, len(respMap), "expected respMap to be empty, but has %d elements", len(respMap))
}

func Test_BulkCountFollowers_WithOneUserIds_ReturnsOneCount(t *testing.T) {
	mockClient := &fakeFollowsCountClient{CountsForUserID: map[string]int32{
		"1": int32(111),
	}}

	userIDs := []string{"1"}
	respMap, err := followsrpc.BulkCountFollowers(ctx, mockClient, userIDs)
	assert.NoError(t, err)
	assert.Equal(t, 1, len(respMap), "expected respMap to have only one element, but found %d", len(respMap))
	assert.Equal(t, int32(111), respMap["1"], "expected count 111 for user 1")
}

func Test_BulkCountFollowers_WithTwoUserIds_ReturnsMultipeCounts(t *testing.T) {
	mockClient := &fakeFollowsCountClient{CountsForUserID: map[string]int32{
		"1": int32(111),
		"2": int32(222),
	}}

	userIDs := []string{"1", "2"}
	respMap, err := followsrpc.BulkCountFollowers(ctx, mockClient, userIDs)
	assert.NoError(t, err)
	assert.Equal(t, 2, len(respMap), "expected respMap to have two elements, but found %d", len(respMap))
	assert.Equal(t, int32(111), respMap["1"], "expected count 111 for user 1")
	assert.Equal(t, int32(222), respMap["2"], "expected count 222 for user 2")
}

func Test_BulkCountFollowers_WithBackendErrors_ReturnsError(t *testing.T) {
	mockClient := &fakeFollowsCountClient{CountsForUserID: map[string]int32{
		"1": int32(111),
	}, ErrorsForUserID: map[string]error{
		"2": twirp.InternalError("error for user 2"),
	}}

	userIDs := []string{"1", "2"}
	respMap, err := followsrpc.BulkCountFollowers(ctx, mockClient, userIDs)
	assert.EqualError(t, err, "twirp error internal: error for user 2", "expected error to be the last client call error")
	assert.Nil(t, respMap, "expected respMap to be nil, but found %v", respMap)
}

func Test_BulkCountFollowers_WithClientPanic_ReturnsInternalError(t *testing.T) {
	mockClient := &panicFollowsCountClient{}

	userIDs := []string{"1", "2"}
	respMap, err := followsrpc.BulkCountFollowers(ctx, mockClient, userIDs)
	assert.EqualError(t, err, "twirp error internal: panic: they are coming!!!")
	assert.Equal(t, 0, len(respMap), "expected respMap to be empty, but has %d elements", len(respMap))
}

// BulkCountFollows tests

func Test_BulkCountFollows_WithEmptyUserIds_ReturnsEmptyList(t *testing.T) {
	mockClient := &fakeFollowsCountClient{}

	userIDs := []string{} // empty list
	respMap, err := followsrpc.BulkCountFollows(ctx, mockClient, userIDs)
	assert.NoError(t, err)
	assert.Equal(t, 0, len(respMap), "expected respMap to be empty, but has %d elements", len(respMap))
}

func Test_BulkCountFollowers_WithOneUserID_ReturnsOneCount(t *testing.T) {
	mockClient := &fakeFollowsCountClient{CountsForUserID: map[string]int32{
		"1": int32(111),
	}}

	userIDs := []string{"1"}
	respMap, err := followsrpc.BulkCountFollows(ctx, mockClient, userIDs)
	assert.NoError(t, err)
	assert.Equal(t, 1, len(respMap), "expected respMap to have only one element, but found %d", len(respMap))
	assert.Equal(t, int32(111), respMap["1"], "expected count 111 for user 1")
}

func Test_BulkCountFollows_WithTwoUserIDs_ReturnsMultipeCounts(t *testing.T) {
	mockClient := &fakeFollowsCountClient{CountsForUserID: map[string]int32{
		"1": int32(111),
		"2": int32(222),
	}}

	userIDs := []string{"1", "2"}
	respMap, err := followsrpc.BulkCountFollows(ctx, mockClient, userIDs)
	assert.NoError(t, err)
	assert.Equal(t, 2, len(respMap), "expected respMap to have two elements, but found %d", len(respMap))
	assert.Equal(t, int32(111), respMap["1"], "expected count 111 for user 1")
	assert.Equal(t, int32(222), respMap["2"], "expected count 222 for user 2")
}

func Test_BulkCountFollows_WithBackendErrors_ReturnsError(t *testing.T) {
	mockClient := &fakeFollowsCountClient{CountsForUserID: map[string]int32{
		"1": int32(111),
	}, ErrorsForUserID: map[string]error{
		"2": twirp.InternalError("error for user 2"),
	}}

	userIDs := []string{"1", "2"}
	respMap, err := followsrpc.BulkCountFollows(ctx, mockClient, userIDs)
	assert.EqualError(t, err, "twirp error internal: error for user 2", "expected error to be the last client call error")
	assert.Nil(t, respMap, "expected respMap to be nil, but found %v", respMap)
}

func Test_BulkCountFollows_WithClientPanic_ReturnsInternalError(t *testing.T) {
	mockClient := &panicFollowsCountClient{}

	userIDs := []string{"1", "2"}
	respMap, err := followsrpc.BulkCountFollows(ctx, mockClient, userIDs)
	assert.EqualError(t, err, "twirp error internal: panic: they are coming!!!")
	assert.Equal(t, 0, len(respMap), "expected respMap to be empty, but has %d elements", len(respMap))
}

// Helpers

// Fake followsCountClient for BulkCountFollowers, can be configured to respond counts by userId
type fakeFollowsCountClient struct {
	CountsForUserID map[string]int32 // configure success response counts per userId
	ErrorsForUserID map[string]error // configure failed response errors per userId
}

func (c *fakeFollowsCountClient) CountFollowers(ctx context.Context, req *followsrpc.UserIDReq) (*followsrpc.CountResp, error) {
	if count, ok := c.CountsForUserID[req.UserId]; ok {
		return &followsrpc.CountResp{Count: count}, nil
	}
	if err, ok := c.ErrorsForUserID[req.UserId]; ok {
		return nil, err
	}
	return &followsrpc.CountResp{Count: int32(0)}, nil
}

func (c *fakeFollowsCountClient) CountFollows(ctx context.Context, req *followsrpc.UserIDReq) (*followsrpc.CountResp, error) {
	if count, ok := c.CountsForUserID[req.UserId]; ok {
		return &followsrpc.CountResp{Count: count}, nil
	}
	if err, ok := c.ErrorsForUserID[req.UserId]; ok {
		return nil, err
	}
	return &followsrpc.CountResp{Count: int32(0)}, nil
}

// Fake followsCountClient for BulkCountFollowers that panics on every call
type panicFollowsCountClient struct{}

func (c *panicFollowsCountClient) CountFollowers(ctx context.Context, req *followsrpc.UserIDReq) (*followsrpc.CountResp, error) {
	panic("they are coming!!!")
}

func (c *panicFollowsCountClient) CountFollows(ctx context.Context, req *followsrpc.UserIDReq) (*followsrpc.CountResp, error) {
	panic("they are coming!!!")
}
