package followsrpcserver_test

import (
	"context"
	"errors"

	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/stretchr/testify/mock"
)

func (s *FollowsSuite) TestBatchHideFollowsSuccess() {
	s.backendMock.On("BulkUpdateFollows", mock.Anything, "1", 10, backend.FollowsKind, backend.HiddenFollowsKind).
		Return(true, nil)

	resp, err := s.followsService.BatchHideFollows(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Equal(true, resp.More)
}

func (s *FollowsSuite) TestBatchRestoreFollowsSuccess() {
	s.backendMock.On("BulkUpdateFollows", mock.Anything, "1", 10, backend.HiddenFollowsKind, backend.FollowsKind).
		Return(true, nil)

	resp, err := s.followsService.BatchRestoreFollows(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Equal(true, resp.More)
}

func (s *FollowsSuite) TestBatchHideFollowersSuccess() {
	s.backendMock.On("BulkUpdateFollows", mock.Anything, "1", 10, backend.FollowersKind, backend.HiddenFollowersKind).
		Return(true, nil)

	resp, err := s.followsService.BatchHideFollowers(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Equal(true, resp.More)
}

func (s *FollowsSuite) TestBatchRestoreFollowersSuccess() {
	s.backendMock.On("BulkUpdateFollows", mock.Anything, "1", 10, backend.HiddenFollowersKind, backend.FollowersKind).
		Return(true, nil)

	resp, err := s.followsService.BatchRestoreFollowers(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Equal(true, resp.More)
}

func (s *FollowsSuite) TestBatchRestoreFollowersFailure() {
	s.backendMock.On("BulkUpdateFollows", mock.Anything, "1", 10, backend.HiddenFollowersKind, backend.FollowersKind).
		Return(false, errors.New("err"))

	_, err := s.followsService.BatchRestoreFollowers(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Error(err)
}

func (s *FollowsSuite) TestBatchDeleteFollowsSuccess() {
	s.backendMock.On("BulkDeleteFollows", mock.Anything, "1", 10, backend.FollowsKind).
		Return(true, nil)

	resp, err := s.followsService.BatchDeleteFollows(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Equal(true, resp.More)
}

func (s *FollowsSuite) TestBatchDeleteFollowersSuccess() {
	s.backendMock.On("BulkDeleteFollows", mock.Anything, "1", 10, backend.FollowersKind).
		Return(true, nil)

	resp, err := s.followsService.BatchDeleteFollowers(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Equal(true, resp.More)
}

func (s *FollowsSuite) TestBatchDeleteFollowersFailure() {
	s.backendMock.On("BulkDeleteFollows", mock.Anything, "1", 10, backend.FollowersKind).
		Return(false, errors.New("err"))

	_, err := s.followsService.BatchDeleteFollowers(context.Background(), &followsrpc.BatchUpdateReq{
		UserId: "1",
		Limit:  10,
	})

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Error(err)
}
