package followsrpcserver_test

import (
	"context"
	"errors"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/feeds/following-service/rpc/followsrpc"
)

func (s *FollowsSuite) TestCountFollowsSuccess() {
	s.backendMock.On("CountFollows", mock.Anything, "1").Return(22, nil)

	resp, err := s.DoCountFollows("1")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal(int32(22), resp.Count, "expect response count to be 22")
}

func (s *FollowsSuite) TestCountFollowsBackendServerError() {
	s.backendMock.On("CountFollows", mock.Anything, "1").
		Return(0, errors.New("trololol")) // oops, something went wrong

	resp, err := s.DoCountFollows("1")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error internal: trololol")
}

func (s *FollowsSuite) TestCountFollowsEmptyUserIDError() {
	resp, err := s.DoCountFollows("")

	s.backendMock.AssertNotCalled(s.T(), "CountFollows") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: user_id is required")
}

// Helpers

func (s *FollowsSuite) DoCountFollows(userID string) (*followsrpc.CountResp, error) {
	req := &followsrpc.UserIDReq{UserId: userID}
	return s.followsService.CountFollows(context.Background(), req)
}
