package followsrpcserver

import (
	"log"
	"strings"
	"time"

	"context"

	pb "code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/twitchtv/twirp"
)

// DeleteFollow deletes the follow relationship between from_user_id and target_user_id. It
// should return a NotFoundError if the follow does not exist.
func (s *Server) DeleteFollow(ctx context.Context, req *pb.FollowReq) (*pb.FollowResp, error) {
	req.FromUserId = strings.TrimSpace(req.FromUserId)
	req.TargetUserId = strings.TrimSpace(req.TargetUserId)
	if req.FromUserId == "" {
		return nil, twirp.RequiredArgumentError("from_user_id")
	}
	if req.TargetUserId == "" {
		return nil, twirp.RequiredArgumentError("target_user_id")
	}
	if err := validateUserID(req.FromUserId); err != nil {
		return nil, twirp.InvalidArgumentError("from_user_id", err.Error())
	}
	if err := validateUserID(req.TargetUserId); err != nil {
		return nil, twirp.InvalidArgumentError("target_user_id", err.Error())
	}
	if req.TargetUserId == req.FromUserId {
		return nil, twirp.InvalidArgumentError("target_user_id", "can not be from_user_id")
	}

	gdbResp, err := s.Backend.GetFollow(ctx, req.FromUserId, req.TargetUserId)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}
	if gdbResp == nil || gdbResp.Edge == nil {
		log.Printf("DeleteFollow - no existing follow FromUserID %v TargetUserId %v \n", req.FromUserId, req.TargetUserId)
		return nil, twirp.NotFoundError("Follow not found")
	}

	err = s.Backend.UnfollowUser(ctx, req.FromUserId, req.TargetUserId)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	t := time.Unix(gdbResp.Edge.Data.CreatedAt.Seconds, int64(gdbResp.Edge.Data.CreatedAt.Nanos))
	return &pb.FollowResp{
		FromUserId:         gdbResp.Edge.Edge.From.Id,
		TargetUserId:       gdbResp.Edge.Edge.To.Id,
		FollowedAt:         t.Format(time.RFC3339),
		BlockNotifications: isBlockingNotification(gdbResp.Edge),
	}, nil
}
