package followsrpcserver_test

import (
	"context"
	"errors"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/feeds/following-service/rpc/followsrpc"
)

func (s *FollowsSuite) TestDeleteFollowSuccess() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(buildBackendGetFollowResponse("1", "2", true), nil)
	s.backendMock.On("UnfollowUser", mock.Anything, "1", "2").Return(nil)

	resp, err := s.DoDeleteFollow("1", "2")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal("1", resp.FromUserId, "expect FromUserId to be 1")
	s.Assert().Equal("2", resp.TargetUserId, "expect TargetUserId to be 2")
	s.Assert().Equal(true, resp.BlockNotifications, "expect BlockNotifications to be true")
}

func (s *FollowsSuite) TestDeleteFollowNotFound() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(nil, nil)

	resp, err := s.DoDeleteFollow("1", "2")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().EqualError(err, "twirp error not_found: Follow not found")
	s.Assert().Nil(resp, "expect resp to be nil")
}

func (s *FollowsSuite) TestDeleteFollowGetFollowFailure() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(nil, errors.New("graphdb timed out"))

	resp, err := s.DoDeleteFollow("1", "2")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().EqualError(err, "twirp error internal: graphdb timed out")
	s.Assert().Nil(resp, "expect resp to be nil")
}

func (s *FollowsSuite) TestDeleteFollowUnfollowUserFailure() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(buildBackendGetFollowResponse("1", "2", true), nil)
	s.backendMock.On("UnfollowUser", mock.Anything, "1", "2").Return(errors.New("graphdb timed out"))

	resp, err := s.DoDeleteFollow("1", "2")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().EqualError(err, "twirp error internal: graphdb timed out")
	s.Assert().Nil(resp, "expect resp to be nil")
}

func (s *FollowsSuite) TestDeleteFollowEmptyFromUserIDErrors() {
	resp, err := s.DoDeleteFollow("", "2")

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: from_user_id is required")
}

func (s *FollowsSuite) TestDeleteFollowEmptyTargetUserIDErrors() {
	resp, err := s.DoDeleteFollow("1", "")

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: target_user_id is required")
}

func (s *FollowsSuite) DoDeleteFollow(fromUserID string, targetUserID string) (*followsrpc.FollowResp, error) {
	req := &followsrpc.FollowReq{FromUserId: fromUserID, TargetUserId: targetUserID}
	return s.followsService.DeleteFollow(context.Background(), req)
}
