package followsrpcserver

import (
	"strings"
	"time"

	"context"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	pb "code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/twitchtv/twirp"
)

// GetBatchFollows loads data about a user's follow association with a list of users
func (s *Server) GetBatchFollows(ctx context.Context, req *pb.BatchFollowsReq) (*pb.BatchFollowsResp, error) {
	req.FromUserId = strings.TrimSpace(req.FromUserId)

	// validate
	if req.FromUserId == "" {
		return nil, twirp.RequiredArgumentError("from_user_id")
	}
	if len(req.TargetUserIds) == 0 {
		return nil, twirp.RequiredArgumentError("target_user_ids")
	}
	if len(req.TargetUserIds) > 100 {
		return nil, twirp.InvalidArgumentError("target_user_ids", "is out of range, limit is 100")
	}

	gdbResp, err := s.Backend.GetBatchFollows(ctx, req.FromUserId, req.TargetUserIds)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	return convertGetBatchFollowsResp(gdbResp), nil
}

func convertGetBatchFollowsResp(batchResp []*graphdbFulton.EdgeGetResponse) *pb.BatchFollowsResp {
	follows := []*pb.FollowResp{}
	for _, resp := range batchResp {
		follows = append(follows, convertLoadedEdge(resp.GetEdge()))
	}
	return &pb.BatchFollowsResp{
		Follows: follows,
	}
}

func convertLoadedEdge(edge *graphdbFulton.LoadedEdge) *pb.FollowResp {
	t := time.Unix(edge.Data.CreatedAt.Seconds, int64(edge.Data.CreatedAt.Nanos))

	return &pb.FollowResp{
		FromUserId:         edge.Edge.From.Id,
		TargetUserId:       edge.Edge.To.Id,
		FollowedAt:         t.Format(time.RFC3339),
		BlockNotifications: isBlockingNotification(edge),
	}
}
