package followsrpcserver_test

import (
	"context"
	"errors"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/mock"
)

func (s *FollowsSuite) TestBatchFollows() {
	userID := "1"
	targetUserIDs := []string{"7489543", "447282"}
	mockResp := buildBackendGetBatchFollowsResp(userID)
	s.backendMock.On("GetBatchFollows", mock.Anything, userID, targetUserIDs).Return(mockResp, nil)

	resp, err := s.DoGetBatchFollows(userID, targetUserIDs)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal(len(mockResp), len(resp.Follows))
}

func (s *FollowsSuite) TestBatchFollowsBackendServerError() {
	userID := "1"
	targetUserIDs := []string{"7489543", "447282"}
	s.backendMock.On("GetBatchFollows", mock.Anything, userID, targetUserIDs).Return(nil, errors.New("maybe next time"))

	resp, err := s.DoGetBatchFollows(userID, targetUserIDs)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error internal: maybe next time")
}

func (s *FollowsSuite) TestBatchFollowsLowerLimitError() {
	userID := "1"
	targetUserIDs := []string{}
	req := &followsrpc.BatchFollowsReq{FromUserId: userID, TargetUserIds: targetUserIDs}
	resp, err := s.followsService.GetBatchFollows(context.Background(), req)

	s.backendMock.AssertNotCalled(s.T(), "GetBatchFollows")
	s.Assert().Nil(resp, "resp should be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: target_user_ids is required")
}

func (s *FollowsSuite) TestBatchFollowsNilTargetUserIDsParamError() {
	userID := "1"
	var targetUserIDs []string
	req := &followsrpc.BatchFollowsReq{FromUserId: userID, TargetUserIds: targetUserIDs}
	resp, err := s.followsService.GetBatchFollows(context.Background(), req)

	s.backendMock.AssertNotCalled(s.T(), "GetBatchFollows")
	s.Assert().Nil(resp, "resp should be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: target_user_ids is required")
}

func (s *FollowsSuite) TestBatchFollowsUserIDError() {
	userID := ""
	targetUserIDs := []string{}
	req := &followsrpc.BatchFollowsReq{FromUserId: userID, TargetUserIds: targetUserIDs}
	resp, err := s.followsService.GetBatchFollows(context.Background(), req)

	s.backendMock.AssertNotCalled(s.T(), "GetBatchFollows")
	s.Assert().Nil(resp, "resp should be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: from_user_id is required")
}

func (s *FollowsSuite) TestBatchFollowsUpperLimitError() {
	userID := "1"
	targetUserIDs := make([]string, 101)
	req := &followsrpc.BatchFollowsReq{FromUserId: userID, TargetUserIds: targetUserIDs}
	resp, err := s.followsService.GetBatchFollows(context.Background(), req)

	s.backendMock.AssertNotCalled(s.T(), "GetBatchFollows")
	s.Assert().Nil(resp, "resp should be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: target_user_ids is out of range, limit is 100")
}

// Helpers

func buildBackendGetBatchFollowsResp(userID string) []*graphdbFulton.EdgeGetResponse {
	createdAt, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		return nil
	}
	return []*graphdbFulton.EdgeGetResponse{
		&graphdbFulton.EdgeGetResponse{
			Edge: &graphdbFulton.LoadedEdge{
				Edge: &graphdbFulton.Edge{
					From: &graphdbFulton.Node{
						Type: backend.UserKind,
						Id:   userID,
					},
					To: &graphdbFulton.Node{
						Type: backend.UserKind,
						Id:   "1234",
					},
					Type: backend.FollowsKind,
				},
				Data: &graphdbFulton.Data{
					CreatedAt: createdAt,
					Data: &graphdbFulton.DataBag{
						Bools: map[string]bool{
							"block_notifications": false,
						},
					},
				},
			},
		},
		&graphdbFulton.EdgeGetResponse{
			Edge: &graphdbFulton.LoadedEdge{
				Edge: &graphdbFulton.Edge{
					From: &graphdbFulton.Node{
						Type: backend.UserKind,
						Id:   userID,
					},
					To: &graphdbFulton.Node{
						Type: backend.UserKind,
						Id:   "12345",
					},
					Type: backend.FollowsKind,
				},
				Data: &graphdbFulton.Data{
					CreatedAt: createdAt,
					Data: &graphdbFulton.DataBag{
						Bools: map[string]bool{
							"block_notifications": false,
						},
					},
				},
			},
		},
	}
}

func (s *FollowsSuite) DoGetBatchFollows(userID string, targetUserIDs []string) (*followsrpc.BatchFollowsResp, error) {
	req := &followsrpc.BatchFollowsReq{
		FromUserId:    userID,
		TargetUserIds: targetUserIDs,
	}
	return s.followsService.GetBatchFollows(context.Background(), req)
}
