package followsrpcserver

import (
	"strings"
	"time"

	"context"

	pb "code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/twitchtv/twirp"
)

// GetFollow loads info about a user following another user
func (s *Server) GetFollow(ctx context.Context, req *pb.FollowReq) (*pb.FollowResp, error) {
	req.FromUserId = strings.TrimSpace(req.FromUserId)
	req.TargetUserId = strings.TrimSpace(req.TargetUserId)

	// validate
	if req.FromUserId == "" {
		return nil, twirp.RequiredArgumentError("from_user_id")
	}
	if req.TargetUserId == "" {
		return nil, twirp.RequiredArgumentError("target_user_id")
	}
	if req.FromUserId == req.TargetUserId {
		return nil, twirp.NotFoundError("Follow not found")
	}

	gdbResp, err := s.Backend.GetFollow(ctx, req.FromUserId, req.TargetUserId)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}
	if gdbResp == nil || gdbResp.Edge == nil {
		return nil, twirp.NotFoundError("Follow not found")
	}

	t := time.Unix(gdbResp.Edge.Data.CreatedAt.Seconds, int64(gdbResp.Edge.Data.CreatedAt.Nanos))
	return &pb.FollowResp{
		FromUserId:         gdbResp.Edge.Edge.From.Id,
		TargetUserId:       gdbResp.Edge.Edge.To.Id,
		FollowedAt:         t.Format(time.RFC3339),
		BlockNotifications: isBlockingNotification(gdbResp.Edge),
	}, nil
}
