package followsrpcserver_test

import (
	"context"
	"errors"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/mock"
)

func (s *FollowsSuite) TestGetFollowSuccess() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").
		Return(buildBackendGetFollowResponse("1", "2", false), nil)

	resp, err := s.DoGetFollow("1", "2")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal("1", resp.FromUserId, "expect FromUserId to be 1")
	s.Assert().Equal("2", resp.TargetUserId, "expect TargetUserId to be 2")
	s.Assert().Equal(false, resp.BlockNotifications, "expect BlockNotifications to be false")
}

func (s *FollowsSuite) TestGetFollowNotFound() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").
		Return(nil, nil) // "1" is not following "2"

	resp, err := s.DoGetFollow("1", "2")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error not_found: Follow not found")
}

func (s *FollowsSuite) TestGetFollowBackendServerError() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").
		Return(nil, errors.New("trololol")) // oops, something went wrong

	resp, err := s.DoGetFollow("1", "2")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error internal: trololol")
}

func (s *FollowsSuite) TestGetFollowEmptyFromUserIDError() {
	resp, err := s.DoGetFollow("", "2")

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: from_user_id is required")
}

func (s *FollowsSuite) TestGetFollowEmptyTargetUserIDError() {
	resp, err := s.DoGetFollow("1", "")

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: target_user_id is required")
}

func (s *FollowsSuite) TestGetFollowSameUserIDError() {
	resp, err := s.DoGetFollow("1", "1")

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error not_found: Follow not found")
}

// Helpers

func buildBackendGetFollowResponse(fromUserID string, targetUserID string, blockNotifications bool) *graphdbFulton.EdgeGetResponse {
	createdAt, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		return nil
	}
	return &graphdbFulton.EdgeGetResponse{
		Edge: &graphdbFulton.LoadedEdge{
			Edge: &graphdbFulton.Edge{
				From: &graphdbFulton.Node{
					Type: backend.UserKind,
					Id:   fromUserID,
				},
				To: &graphdbFulton.Node{
					Type: backend.UserKind,
					Id:   targetUserID,
				},
				Type: backend.FollowsKind,
			},
			Data: &graphdbFulton.Data{
				CreatedAt: createdAt,
				Data: &graphdbFulton.DataBag{
					Bools: map[string]bool{
						"block_notifications": blockNotifications,
					},
				},
			},
		},
	}
}

func (s *FollowsSuite) DoGetFollow(fromUserID string, targetUserID string) (*followsrpc.FollowResp, error) {
	req := &followsrpc.FollowReq{FromUserId: fromUserID, TargetUserId: targetUserID}
	return s.followsService.GetFollow(context.Background(), req)
}
