package followsrpcserver

import (
	"log"
	"strings"
	"time"

	"context"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	pb "code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/twitchtv/twirp"
)

// ListFollowers that are following the user
func (s *Server) ListFollowers(ctx context.Context, req *pb.FollowsReq) (*pb.FollowsResp, error) {
	req.UserId = strings.TrimSpace(req.UserId)
	if req.UserId == "" {
		return nil, twirp.RequiredArgumentError("user_id")
	}

	if req.Limit < 0 {
		return nil, twirp.InvalidArgumentError("limit", "can not be negative")
	}
	if req.Limit == 0 {
		req.Limit = followsReqDefaultLimit
		err := s.Stats.Inc("listfollowers.length_0", 1, 0.1)
		if err != nil {
			log.Print(err)
		}
	}

	if req.Offset > 0 {
		err := s.Stats.Inc("offset.used", 1, 1.0)
		if err != nil {
			log.Print(err)
		}
	}
	err := s.Stats.TimingDuration("listfollowers.req_limit", time.Duration(req.Limit)*time.Millisecond, 0.1)
	if err != nil {
		log.Print(err)
	}

	direction := strings.ToLower(req.OrderByFollowedAt.String()) // "asc" (default) or "desc"

	gdbResp, err := s.Backend.ListFollowers(ctx, req.UserId, req.Cursor, int(req.Limit), int(req.Offset), direction)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	// Cache-Control header, used by Visage's cache mechanism
	err = twirp.SetHTTPResponseHeader(ctx, "Cache-Control", "public, max-age=60")
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	return convertFollowersResp(gdbResp), nil
}

func convertFollowersResp(response *graphdbFulton.EdgeListResponse) *pb.FollowsResp {
	followers := []*pb.FollowResp{}
	for _, edge := range response.Edges {
		followers = append(followers, convertFollowers(edge))
	}
	return &pb.FollowsResp{
		Follows: followers,
		Cursor:  response.Cursor,
	}
}

func convertFollowers(edge *graphdbFulton.LoadedCursoredEdge) *pb.FollowResp {
	t := time.Unix(edge.Edge.Data.CreatedAt.Seconds, int64(edge.Edge.Data.CreatedAt.Nanos))

	return &pb.FollowResp{
		FromUserId:         edge.Edge.Edge.To.Id,
		TargetUserId:       edge.Edge.Edge.From.Id,
		FollowedAt:         t.Format(time.RFC3339),
		BlockNotifications: isBlockingNotification(edge.Edge),
		Cursor:             edge.Cursor,
	}
}
