package followsrpcserver_test

import (
	"context"
	"errors"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/mock"
)

func (s *FollowsSuite) TestListFollowersSuccess() {
	mockResp := buildBackendListFollowersResp("1")
	s.backendMock.On("ListFollowers", mock.Anything, "1", "", defaultLimit, defaultOffset, defaultOrderByFollowedAt).
		Return(mockResp, nil) // followers of "1"

	resp, err := s.DoListFollowers("1")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal(len(mockResp.Edges), len(resp.Follows), "expect Follows to be populated with data")
	s.Assert().Equal(mockResp.Cursor, resp.Cursor, "expect Cursor to have the same value as the backend response")
}

func (s *FollowsSuite) TestListFollowersBackendServerError() {
	s.backendMock.On("ListFollowers", mock.Anything, "1", "", defaultLimit, defaultOffset, defaultOrderByFollowedAt).
		Return(nil, errors.New("trololol")) // oops, something went wrong

	resp, err := s.DoListFollowers("1")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error internal: trololol")
}

func (s *FollowsSuite) TestListFollowersWithPaginationSuccess() {
	cursor := "cursor-kakaka"
	limit := 33
	offset := 44
	direction := "desc"
	mockResp := buildBackendListFollowersResp("1")
	s.backendMock.On("ListFollowers", mock.Anything, "1", cursor, limit, offset, direction).
		Return(mockResp, nil)

	req := &followsrpc.FollowsReq{UserId: "1", Cursor: cursor, Limit: int32(limit), Offset: int32(offset), OrderByFollowedAt: followsrpc.FollowsReq_DESC}
	resp, err := s.followsService.ListFollowers(context.Background(), req)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal(len(mockResp.Edges), len(resp.Follows), "expect Follows to be populated with data")
	s.Assert().Equal(mockResp.Cursor, resp.Cursor, "expect Cursor to have the same value as the backend response")
}

func (s *FollowsSuite) TestListFollowersNegativeLimitError() {
	req := &followsrpc.FollowsReq{UserId: "1", Limit: -1}
	resp, err := s.followsService.ListFollowers(context.Background(), req)

	s.backendMock.AssertNotCalled(s.T(), "ListFollowers") // validation should fail before making backend requests
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: limit can not be negative")
}

func (s *FollowsSuite) TestListFollowersEmptyUserIDError() {
	resp, err := s.DoListFollowers("")

	s.backendMock.AssertNotCalled(s.T(), "ListFollowers") // validation should fail before making backend requests
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: user_id is required")
}

// Helpers

func buildBackendListFollowersResp(userID string) *graphdbFulton.EdgeListResponse {
	createdAt, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		return nil
	}
	return &graphdbFulton.EdgeListResponse{
		Edges: []*graphdbFulton.LoadedCursoredEdge{
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   userID,
						},
						To: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "222",
						},
						Type: backend.FollowersKind,
					},
					Data: &graphdbFulton.Data{
						CreatedAt: createdAt,
						Data: &graphdbFulton.DataBag{
							Bools: map[string]bool{
								"block_notifications": false,
							},
						},
					},
				},
				Cursor: "inner cursor1",
			},
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   userID,
						},
						To: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "333",
						},
						Type: backend.FollowersKind,
					},
					Data: &graphdbFulton.Data{
						CreatedAt: createdAt,
						Data: &graphdbFulton.DataBag{
							Bools: map[string]bool{
								"block_notifications": false,
							},
						},
					},
				},
				Cursor: "inner cursor2",
			},
		},
		Cursor: "A Cursor!",
	}
}

func (s *FollowsSuite) DoListFollowers(userID string) (*followsrpc.FollowsResp, error) {
	req := &followsrpc.FollowsReq{UserId: userID} // other values are defaults
	return s.followsService.ListFollowers(context.Background(), req)
}
